/*!
 * @file shell_buffer.h
 * @brief The Shell buffer Tasks
 * @see debug.h
*/

#ifndef _SHELL_BUFFER_H
#define _SHELL_BUFFER_H

#include <list>

#define      SHELL_BUFFER_SIZE       16384         //!< The Size of the input-buffers (should be large enough to carry any kind of input)

namespace OrxShell
{
  // FORWARD DECLARATION
  class Shell;

  //! A class handling output from orxonox via debug.h
  /**
   * the ShellBuffer redirects output from PRINTF(x) to the Shell and STDOUT
   * the ShellBuffer is a front-filling queue of limited length, that has the
   * youngest added Entry at the beginning, and the oldest at the end.
   */
  class ShellBuffer
  {

  public:
    virtual ~ShellBuffer();
    /** @returns a Pointer to the only object of this Class */
    inline static ShellBuffer* getInstance() { if (!ShellBuffer::singletonRef) ShellBuffer::singletonRef = new ShellBuffer();  return ShellBuffer::singletonRef; };
    /** @returns true if this class is instanciated, false otherwise */
    inline static bool isInstanciated() { return (ShellBuffer::singletonRef == NULL)?false:true; };

    void registerShell(Shell* shell);
    void unregisterShell(Shell* shell);

    static bool addBufferLineStatic(const char* line, ...);
    void addBufferLine(const char* line, va_list arg);

    // BUFFER //
    /** @param bufferSize the new Buffer-Size */
    void setBufferSize(unsigned int bufferSize) { this->bufferSize = bufferSize; };
    void flush();

    /** @returns the List of stings from the Buffer */
    const std::list<std::string>& getBuffer() const { return this->buffer; };
    /** @returns the Count of lines processed by the Shell. */
    inline unsigned long getLineCount() const { return this->lineCount; };

    void debug() const;

  private:
    ShellBuffer();

  private:
    static ShellBuffer*           singletonRef;                       //!< The singleton-reference to the only memeber of this class.
    unsigned int                  bufferSize;                         //!< The Size of the buffer

    Shell*                        shell;                              //!< the Registered Shell.
    char                          bufferArray[SHELL_BUFFER_SIZE];     //!< a BUFFER for fast writing
    char                          keepBufferArray[SHELL_BUFFER_SIZE]; //!< a BUFFER to have multi-non-newLine commands be copied into the shell.
    bool                          keepBuffer;                         //!< if the keepbuffer contains unfinished lines.

    unsigned long                 lineCount;                          //!< how many Lines have been written out so far.

    // The Beginning of buffer (buffer.front()) is the last added line.
    static std::list<std::string> buffer;                             //!< A list of stored char-arrays(strings) to store the history
  };

}

#endif /* _SHELL_BUFFER_H */
