/*!
 * @file shell_command.h
 * Definition of a on-screen-shell
 */

#ifndef _SHELL_COMMAND_H
#define _SHELL_COMMAND_H

#include "base_object.h"

#include "executor/executor.h"
#include "shell_completion_plugin.h"

#define     SHELL_COMMAND_MAX_SIZE      //!< The maximum size of a Shell Command

namespace OrxShell
{
  // FORWARD DECLARATION
  class ShellCommandClass;
  class ShellCommandAlias;
  class CompletorPlugin;

  /**
   * @brief an easy to use Macro to create a Command
   * @param command the name of the command (without "" around the string)
   * @param class the name of the class to apply this command to (without the "" around the string)
   * @param function the function to call
   *
   * MEANING:
   *  ShellCommand* someUniqueVarName =
   *       ShellCommand<ClassName>::registerCommand("commandNameInShell", "ClassName", &ClassName::FunctionToCall);
   *
   * In the Shell you would call this Command using:
   * $ ClassName [ObjectName] commandNameInShell [parameters]
   */
  //#define SHELL_COMMAND(command, class, function) \
  //        ShellCommand* shell_command_##class##_##command = ShellCommand<class>::registerCommand(#command, #class, &class::function)
#define SHELL_COMMAND(command, class, function) \
           OrxShell::ShellCommand* shell_command_##class##_##command = OrxShell::ShellCommand::registerCommand(#command, #class, createExecutor<class>(&class::function))

  /**
   * @brief an easy to use Macro to create a Command
   * @param command the name of the command (without "" around the string)
   * @param class the name of the class to apply this command to (without the "" around the string)
   * @param function the function to call
   *
   * MEANING:
   *  ShellCommand* someUniqueVarName =
   *       ShellCommand<ClassName>::registerCommand("commandNameInShell", "ClassName", &ClassName::FunctionToCall);
   *
   * In the Shell you would call this Command using:
   * $ ClassName [ObjectName] commandNameInShell [parameters]
   */
#define SHELL_COMMAND_STATIC(command, class, function) \
           OrxShell::ShellCommand* shell_command_##class##_##command = OrxShell::ShellCommand::registerCommand(#command, #class, createExecutor<class>(function))



  //! a baseClass for all possible ShellCommands
  class ShellCommand : public BaseObject
  {
    friend class ShellCommandClass;
  public:
    static bool execute (const std::string& executionString);

    ShellCommand* describe(const std::string& description);
    ShellCommand* setAlias(const std::string& alias);
    ShellCommand* defaultValues(const MultiType& value0 = MT_NULL, const MultiType& value1 = MT_NULL,
                                const MultiType& value2 = MT_NULL, const MultiType& value3 = MT_NULL,
                                const MultiType& value4 = MT_NULL);
    ShellCommand* completionPlugin(unsigned int parameter, const CompletorPlugin& completorPlugin);

    static ShellCommand* registerCommand(const std::string& commandName, const std::string& className, Executor* executor);
    static void unregisterCommand(const std::string& commandName, const std::string& className);
    static const ShellCommand* const getCommand(const std::string& commandName, const std::string& className);
    static const ShellCommand* const getCommand(const std::string& commandName, const ShellCommandClass* cmdClass);
    static bool exists(const std::string& commandName, const std::string& className);
    static const ShellCommand* const getCommandFromInput(const std::string& inputLine, unsigned int& paramBegin, std::vector<BaseObject*>* boList = NULL);
    static const ShellCommand* const getCommandFromInput(const SubString& strings, unsigned int& paramBegin, std::vector<BaseObject*>* boList = NULL);

    const ShellCommandClass* const getCommandClass() const { return this->shellClass; };
    const ShellCommandAlias* const getAlias() const { return this->alias; }
    unsigned int getParamCount() const { return this->executor->getParamCount(); }
    const CompletorPlugin* const getCompletorPlugin(unsigned int i) const { return this->completors[i]; };

    static void debug();

  protected:
    ShellCommand(const std::string& commandName, const std::string& className, Executor* executor);
    virtual ~ShellCommand();

    static bool fillObjectList(const std::string& objectDescriptor, const ShellCommand* cmd, std::vector<BaseObject*>* boList);
    static const std::string& paramToString(long parameter);

  private:
    ShellCommandClass*               shellClass;            //!< A Pointer to the Shell-Class this Command belongs to.
    ShellCommandAlias*               alias;                 //!< An Alias for the Class.

    std::string                      description;           //!< A description for this commnand. (initially ""). Assigned with (create)->describe("blablabla");
    std::vector<CompletorPlugin*>    completors;            //!< Completors for the Parameters
    Executor*                        executor;              //!< The Executor, that really executes the Function.
  };

  //! A Class, that handles aliases.
  class ShellCommandAlias
  {
  public:
    ShellCommandAlias(const std::string& aliasName, ShellCommand* command);
    ~ShellCommandAlias();

    /** @returns the Name of the Alias. */
    const std::string& getName() const { return this->aliasName; };
    /** @returns the Command, this Alias is asociated with */
    ShellCommand* getCommand() const { return this->command; };
    static bool getCommandListOfAlias(std::list<std::string>& stringList);
    static const std::vector<ShellCommandAlias*>& getAliases() { return ShellCommandAlias::aliasList; };

  private:
    std::string     aliasName;       //!< the name of the Alias
    ShellCommand*   command;         //!< a pointer to the command, this alias executes.

    static std::vector<ShellCommandAlias*> aliasList; //!< A list of Aliases to A Commands.
  };

}

#endif /* _SHELL_COMMAND_H */
