/*!
 * @file keynames.h
 *  Key/button naming functions
 *
 * Converts strings to SDLK/SDL_BUTTON values and vice versa
 */
#ifndef _KEY_NAMES_H
#define _KEY_NAMES_H

#include <string>

/**
 * @brief converts an EVKey into a String, naming the Event.
 * @param key the Key (either key or button) to convert.
 * @returns the String containing the Event.
 */
std::string EVToKeyName(int key);
/**
 * @brief converts a KeyName into an Event.
 * @param keyName the Key to transform.
 * @returns the Event-Number
 */
int KeyNameToEV(const std::string& keyName);


/**
 * @brief converts a button name string to a integer representing the corresponding SDL mouse button identifier
 * @param name: the name of the mouse button
 * @return an int containing the SDL identifier of the mouse button or -1 if the button name is not valid
 */
int buttonnameToSDLB(const std::string& name);

/**
 * @brief converst a SDL mouse button identifier to a name string
 * @param button: an SDL mouse button identifier
 * @return a pointer to a string containing the name of the mouse button
 */
std::string SDLBToButtonname( int button);

/**
 * @brief converts a key name string to a integer representing the corresponding SDLK sym
 * @param name: the name of the key
 * @return the SDLK sym of the named key or -1 if the key name is not valid
 */
int keynameToSDLK(const std::string& name);

/**
 * @brief converts an SDLK sym to a name string
 * @param key: the SDLK sym
 * @return a pointer to a string containig the name of the key
 */
std::string SDLKToKeyname( int key);

#endif /* _KEY_NAMES_H */
