/*!
 * @file synchronizeable_var.h
 * @brief Definition of SynchronizeableVar
*/

#ifndef _SYNCHRONIZEABLE_VAR_H
#define _SYNCHRONIZEABLE_VAR_H

#include <string>
#include "netdefs.h"
#include <assert.h>

enum {
  PERMISSION_SERVER = 1,
  PERMISSION_OWNER  = 2,
  PERMISSION_ALL    = 4
};

class SynchronizeableVar {

  public:
    SynchronizeableVar( void * ptrIn, void * ptrOut, std::string name, int length, int permission = PERMISSION_SERVER, int priority = 0 );
    virtual ~SynchronizeableVar();

    /**
     * check if synchronizeable wants to be informed on changes
     * @return true if synchronizeable wants to be informed on changes
     */
    inline bool beWatched(){ return this->bWatched; }
    
    /** 
     * set flag if synchronizeable wants to be informed on changes
     */
    inline void setWatched( bool watched ){ this->bWatched = watched; }

    /**
     * write var data to byte buffer
     * @param buf pointer to write to
     * @param maxLength writeToBuf will not write more than maxLength bytes
     * @return number bytes written
     */
    virtual int writeToBuf( byte * buf, int maxLength ) = 0;

    /**
     * read var data from byte buffer
     * @param buf pointer to read from
     * @param maxLength readFromBuf will not read more than maxLength bytes
     * @return number bytes read
     */
    virtual int readFromBuf( byte * buf, int maxLength ) = 0;

    /**
     * check if writeToBuf will return the same size every time
     * @return true if same size every time
     */
    virtual bool hasStaticSize() = 0;

    /**
     * get size writeToBuf needs
     * @return size in bytes
     */
    virtual int getSize(){ return length; }

    /**
     * check for permission to write
     * @return true if you can write
     */
    inline bool checkPermission( int permission ){ return (permission & this->permission) != 0; }

    /**
     * get variable name
     * @return name
     */
    inline std::string getName(){ return name; }

    /**
     * set variable name
     * @param name new name
     */
    inline void setName( std::string name ) { this->name = name; }

    /**
     * get priority
     * @return priority
     */
    inline int getPriority() { return this->priority; }

    /**
     * set priority
     * @param p priority
     */
    inline void setPriority( int p ) { this->priority = p; }

    /**
     * reset priority to variable specific default value
     */
    inline void resetPriority() { this->priority = this->realPriority; }
    
    /**
     * reads actual size from buffer. this is used when size is not constant
     * @param buf pointer to data
     * @param maxLength maximal size of data
     * @return same as readFromBuf would return
     */
    virtual inline int getSizeFromBuf( byte * buf, int maxLength ){ return this->getSize(); }
    
    /**
     * set variable id
     * @param id
     */
    inline void setVarId( int id ){ this->varId = id; }
    
    /** 
     * get variable id
     * @return variable id
     */
    inline int getVarId(){ return varId; }
    
    /**
     * set hasChanged
     * @param changed
     */
    inline void setHasChanged( bool changed ){ this->changed = changed; }
    
    /** 
     * get hasChanged
     * @return variable id
     */
    inline bool getHasChanged(){ return changed; }
    
    /**
     * print out variable value
     */
    virtual void debug() = 0;


  private:
    bool bWatched;      //!< true if synchronizeable wants to be informed on changes
    
    int permission;     //!< who is allowed to change this var
    int priority;       //!< priority assigned to var
    int realPriority;   //!< priority assigned to var, increased every time not sent
    int varId;          //!< id to identify varables
    
    bool changed;       //!< true if last readFromBuf changed anything

  protected:
    void * ptrIn;       //!< pointer to data (read)
    void * ptrOut;      //!< pointer to data (write)
    int length;         //!< data length


    std::string name;    //!< variable name (for debugging)

};

#endif /* _PROTO_CLASS_H */
