/*!
 * @file synchronizeable.h
    \brief interface for all classes that have to be synchronized
 */

#ifndef _SYNCHRONIZEABLE_H
#define _SYNCHRONIZEABLE_H

#include "base_object.h"
#include "netdefs.h"
#include "converter.h"
#include "vector.h"
#include "quaternion.h"
#include "synchronizeable_var/synchronizeable_var.h"
#include "synchronizeable_var/synchronizeable_vector.h"
#include "synchronizeable_var/synchronizeable_quaternion.h"
#include "synchronizeable_var/synchronizeable_string.h"
#include "synchronizeable_var/synchronizeable_int.h"
#include "synchronizeable_var/synchronizeable_float.h"
#include "synchronizeable_var/synchronizeable_bool.h"
#include "synchronizeable_var/synchronizeable_uint.h"


#include <vector>
#include <list>

//State constants: They have to be of the form 2^n
#define STATE_SERVER 1

struct StateHistoryEntry
{
  int             stateId;
  byte *          data;
  int             dataLength;
  std::list<int>  sizeList;
};

typedef std::list<StateHistoryEntry*> StateHistory;

typedef std::vector<StateHistory> UserStateHistory;

typedef std::vector<SynchronizeableVar*> SyncVarList;

class NetworkStream;

class Synchronizeable : virtual public BaseObject
{

  public:
    Synchronizeable();
    virtual ~Synchronizeable();

    void setIsServer( bool isServer );
    bool isServer();

    virtual void varChangeHandler( std::list<int> & id );

    virtual int getStateDiff( int userId, byte* data, int maxLength, int stateId, int fromStateId, int priorityTH );
    virtual int setStateDiff( int userId, byte* data, int length, int stateId, int fromStateId );
    virtual void cleanUpUser( int userId );

    void registerVar( SynchronizeableVar * var );
    int registerVarId( SynchronizeableVar * var );

    inline void setUniqueID( int id ){ uniqueID = id; }
    inline int  getUniqueID() const { return uniqueID; }
    inline int getHostID() { return this->hostID; }

    inline int getOwner(){ return owner; }
    inline void setOwner(int owner){ this->owner = owner; }

    /** @returns true if this Synchronizeable wants to be synchronized over network */
    inline bool beSynchronized() { return this->bSynchronize; }
    /** @param bSynchronize sets the Synchronizeable to be sunchronized or not */
    inline void setSynchronized(bool bSynchronize) { this->bSynchronize = bSynchronize; }

    inline void setNetworkStream(NetworkStream* stream) { this->networkStream = stream; }
    inline NetworkStream* getNetworkStream() { return this->networkStream; }


  protected:
    NetworkStream*    networkStream;  //!< reference network stream we are connected to
    int               state;

  private:
    int               uniqueID;       //!< unique id assigned to synchronizeable
    int               mLeafClassId;   //!< store leafClassId to send via states
    int               owner;          //!< hostId of owner ( 0 if none / server )
    int               hostID;         //!< my own host id
    bool              bSynchronize;   //!< do we need beeing synchronized?

    SyncVarList       syncVarList;    //!< list containing variables to synchronize

    UserStateHistory  sentStates;     //!< store already sent states to create diffs from, offset corresponds to the user id
    UserStateHistory  recvStates;     //!< store recieved states to apply diffs, offset corresponds to the user id

};
#endif /* _SYNCHRONIZEABLE_H */
