/*!
 * @file network_stream.h
 *  implementation of a network pipe
 */

#ifndef _NETWORK_STREAM
#define _NETWORK_STREAM

#include <vector>
#include <list>
#include <map>

#include "data_stream.h"
#include "network_protocol.h"
#include "server_socket.h"
#include "handshake.h"

class Synchronizeable;
class NetworkSocket;
class ServerSocket;
class ConnectionMonitor;
class NetworkProtocol;
class NetworkGameManager;

class PeerInfo
{
  public:
    PeerInfo() { clear(); }
    void clear() { userId = 0; isServer = false; socket = NULL; handshake = NULL; lastAckedState = 0; lastRecvedState = 0; }
    int             userId;
    bool            isServer;
    NetworkSocket * socket;
    Handshake *     handshake;
    int             lastAckedState;
    int             lastRecvedState;
};

typedef std::list<Synchronizeable*>  SynchronizeableList;
typedef std::map<int,PeerInfo>       PeerList;


class NetworkStream : public DataStream
{

  public:
    NetworkStream();
    NetworkStream( std::string host, int port);
    NetworkStream( int port );

    virtual ~NetworkStream();
    void init();

    void createNetworkGameManager();
    void startHandshake();

    void connectSynchronizeable(Synchronizeable& sync);
    void disconnectSynchronizeable(Synchronizeable& sync);

    inline bool isServer() const { return (this->type == NET_SERVER)? true:false; }
    inline bool isActive() const { return this->bActive; }

    inline int getMaxConnections(){ return MAX_CONNECTIONS; }

    virtual void processData();

    inline SynchronizeableList::const_iterator getSyncBegin(){ return synchronizeables.begin(); }
    inline SynchronizeableList::const_iterator getSyncEnd(){ return synchronizeables.end(); }
    int getSyncCount();

    inline bool isUserIdActive( int userID ) { return (peers.find(userID) != peers.end()); }
    inline bool isUserServer( int userID ){ if ( !isUserIdActive(userID) ) return false; return peers[userID].isServer; }

    void debug();
    
    inline PeerInfo & getPeerInfo( int userId ) { return peers[userId]; }


  private:
    void updateConnectionList();
    void handleHandshakes();
    void handleUpstream();
    void handleDownstream(); 


  private:
    NetworkProtocol*           networkProtocol;
    ConnectionMonitor*         connectionMonitor;
    SynchronizeableList        synchronizeables;
    PeerList                   peers;
    ServerSocket*              serverSocket;
    int                        type;
    Header                     packetHeader;
    bool                       bActive;
    std::list<int>             freeSocketSlots;

    int                        myHostId;
    
    int                        currentState;

    NetworkGameManager*        networkGameManager;

    std::map<int,int>          oldSynchronizeables;
};
#endif /* _NETWORK_STREAM */
