/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli
*/

#define DEBUG_SPECIAL_MODULE 3/* DEBUG_MODULE_COLLISION_DETECTION*/

#include "obb_tree_node.h"
#include "obb_tree.h"
#include "obb.h"

#include "matrix.h"
#include "model.h"
#include "world_entity.h"
#include "plane.h"

#include "color.h"
#include "glincl.h"

#include <list>
#include <vector>
#include "debug.h"



using namespace std;


GLUquadricObj* OBBTreeNode_sphereObj = NULL;


/**
 *  standard constructor
 * @param tree: reference to the obb tree
 * @param depth: the depth of the obb tree to generate
 */
OBBTreeNode::OBBTreeNode (const OBBTree& tree, OBBTreeNode* prev, int depth)
    : BVTreeNode()
{
  this->setClassID(CL_OBB_TREE_NODE, "OBBTreeNode");

  this->obbTree = &tree;
  this->nodePrev = prev;
  this->depth = depth;
  this->nextID = 0;

  this->nodeLeft = NULL;
  this->nodeRight = NULL;
  this->bvElement = NULL;

  this->triangleIndexList1 = NULL;
  this->triangleIndexList2 = NULL;

  this->modelInf = NULL;
  this->triangleIndexes = NULL;

  if( OBBTreeNode_sphereObj == NULL)
    OBBTreeNode_sphereObj = gluNewQuadric();

  this->owner = NULL;

  /* debug ids */
  if( this->nodePrev)
    this->treeIndex = 100 * this->depth + this->nodePrev->getID();
  else
    this->treeIndex = 0;
}


/**
 *  standard deconstructor
 */
OBBTreeNode::~OBBTreeNode ()
{
  if( this->nodeLeft)
    delete this->nodeLeft;
  if( this->nodeRight)
    delete this->nodeRight;

  if( this->bvElement)
    delete this->bvElement;

//   if( this->triangleIndexList1 != NULL)
//     delete [] this->triangleIndexList1;
//   if( this->triangleIndexList2 != NULL)
//     delete [] this->triangleIndexList2;
}


/**
 *  creates a new BVTree or BVTree partition
 * @param depth: how much more depth-steps to go: if == 1 don't go any deeper!
 * @param modInfo: model informations from the abstrac model
 *
 * this function creates the Bounding Volume tree from a modelInfo struct and bases its calculations
 * on the triangle informations (triangle soup not polygon soup)
 */
void OBBTreeNode::spawnBVTree(const modelInfo& modelInf, const int* triangleIndexes, int length)
{
  PRINTF(3)("\n==============================Creating OBB Tree Node==================\n");
  PRINT(3)(" OBB Tree Infos: \n");
  PRINT(3)("\tDepth: %i \n\tTree Index: %i \n\tNumber of Triangles: %i\n", depth, this->treeIndex, length);
  this->depth = depth;

  this->bvElement = new OBB();
  this->bvElement->modelInf = &modelInf;
  this->bvElement->triangleIndexes = triangleIndexes;
  this->bvElement->triangleIndexesLength = length;

  /* create the bounding boxes in three steps */
  this->calculateBoxCovariance(*this->bvElement, modelInf, triangleIndexes, length);
  this->calculateBoxEigenvectors(*this->bvElement, modelInf, triangleIndexes, length);
  this->calculateBoxAxis(*this->bvElement, modelInf, triangleIndexes, length);

  /* do we need to descent further in the obb tree?*/
  if( likely( this->depth > 0))
  {
    this->forkBox(*this->bvElement);

    if( this->triangleIndexLength1 >= 3)
    {
      this->nodeLeft = new OBBTreeNode(*this->obbTree, this, depth - 1);
      this->nodeLeft->spawnBVTree(modelInf, this->triangleIndexList1, this->triangleIndexLength1);
    }
    if( this->triangleIndexLength2 >= 3)
    {
      this->nodeRight = new OBBTreeNode(*this->obbTree, this, depth - 1);
      this->nodeRight->spawnBVTree(modelInf, this->triangleIndexList2, this->triangleIndexLength2);
    }
  }
}



/**
 *  calculate the box covariance matrix
 * @param box: reference to the box
 * @param modelInf: the model info structure of the model
 * @param tirangleIndexes: an array with the indexes of the triangles inside this
 * @param length: the length of the indexes array
 */
void OBBTreeNode::calculateBoxCovariance(OBB& box, const modelInfo& modelInf, const int* triangleIndexes, int length)
{
  float     facelet[length];                         //!< surface area of the i'th triangle of the convex hull
  float     face = 0.0f;                             //!< surface area of the entire convex hull
  Vector    centroid[length];                        //!< centroid of the i'th convex hull
  Vector    center;                                  //!< the center of the entire hull
  Vector    p, q, r;                                 //!< holder of the polygon data, much more conveniant to work with Vector than sVec3d
  Vector    t1, t2;                                  //!< temporary values
  float     covariance[3][3] = {0,0,0, 0,0,0, 0,0,0};//!< the covariance matrix
  sVec3D*   tmpVec = NULL;                           //!< a temp saving place for sVec3Ds


  /* fist compute all the convex hull face/facelets and centroids */
  for( int i = 0; i < length ; ++i)
  {
    tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[i]].indexToVertices[0]]);
    p = *tmpVec;
    tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[i]].indexToVertices[1]]);
    q = *tmpVec;
    tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[i]].indexToVertices[2]]);
    r = *tmpVec;

    /* finding the facelet surface via cross-product */
    t1 = p - q;
    t2 = p - r;
    facelet[i] = 0.5f * fabs( t1.cross(t2).len() );
    /* update the entire convex hull surface */
    face += facelet[i];

    /* calculate the cetroid of the hull triangles */
    centroid[i] = (p + q + r) / 3.0f;
    /* now calculate the centroid of the entire convex hull, weighted average of triangle centroids */
    center += centroid[i] * facelet[i];
    /* the arithmetical center */
  }
  /* take the average of the centroid sum */
  center /= face;


  /* now calculate the covariance matrix - if not written in three for-loops,
     it would compute faster: minor */
  for( int j = 0; j < 3; ++j)
  {
    for( int k = 0; k < 3; ++k)
    {
      for( int i = 0; i < length; ++i)
      {
        tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[i]].indexToVertices[0]]);
        p = *tmpVec;
        tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[i]].indexToVertices[1]]);
        q = *tmpVec;
        tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[i]].indexToVertices[2]]);
        r = *tmpVec;

        covariance[j][k] = facelet[i] * (9.0f * centroid[i][j] * centroid[i][k] + p[j] * p[k] +
                           q[j] * q[k] + r[j] * r[k]);
      }
      covariance[j][k] = covariance[j][k] / (12.0f * face) - center[j] * center[k];
    }
  }
  for( int i = 0; i < 3; ++i)
  {
    box.covarianceMatrix[i][0] = covariance[i][0];
    box.covarianceMatrix[i][1] = covariance[i][1];
    box.covarianceMatrix[i][2] = covariance[i][2];
  }
  box.center = center;


  std::vector<int>           vertIndexVector;                           //!< vertex indexes list
  int                        vertIndex;                                 //!< index to vertex
  bool                       vertexFound;                               //!< vertex found flag
  Vector                     arithCenter;                               //!< aritmetical center

  /* calculate the arithmetical center of the box */

  /* go thourgh all vertices, add only the used vertices indexes */
//   for( int i = 0; i < length; ++i)
//   {
//     for(int j = 0; j < 3; ++j)
//     {
//       vertIndex = modelInf.pTriangles[triangleIndexes[i]].indexToVertices[j];
//
//       vertexFound = false;
//       for( int i = 0; i < vertIndexVector.size(); i++)
//       {
//         if( vertIndexVector[i] == vertIndex)
//           vertexFound = true;
//       }
//       if( !vertexFound)
//         vertIndexVector.push_back(vertIndex);
//     }
//   }
//   /* now realy calculate the center */
//   for( int i = 0; i < vertIndexVector.size(); ++i)
//   {
//     tmpVec = (sVec3D*)(&modelInf.pVertices[vertIndexVector[i]]);
//     arithCenter += *tmpVec;
//   }
//   box.arithCenter = arithCenter / vertIndexVector.size();



  /* debug output section*/
  PRINTF(3)("\nOBB Covariance Matrix:\n");
  for(int j = 0; j < 3; ++j)
  {
    PRINT(3)("\t\t");
    for(int k = 0; k < 3; ++k)
    {
      PRINT(3)("%11.4f\t", covariance[j][k]);
    }
    PRINT(3)("\n");
  }
  PRINTF(3)("\nWeighteed OBB Center:\n\t\t%11.4f\t %11.4f\t %11.4f\n", center.x, center.y, center.z);
//   PRINTF(3)("\nArithmetical OBB Center:\n\t\t%11.4f\t %11.4f\t %11.4f\n", box.arithCenter.x, box.arithCenter.y, box.arithCenter.z);

  /* write back the covariance matrix data to the object oriented bouning box */
}



/**
 *  calculate the eigenvectors for the object oriented box
 * @param box: reference to the box
 * @param modelInf: the model info structure of the model
 * @param tirangleIndexes: an array with the indexes of the triangles inside this
 * @param length: the length of the indexes array
 */
void OBBTreeNode::calculateBoxEigenvectors(OBB& box, const modelInfo& modelInf,
    const int* triangleIndexes, int length)
{

  Vector         axis[3];                            //!< the references to the obb axis
  Matrix         covMat(  box.covarianceMatrix  );   //!< covariance matrix (in the matrix dataform)

  /*
  now getting spanning vectors of the sub-space:
  the eigenvectors of a symmertric matrix, such as the
  covarience matrix are mutually orthogonal.
  after normalizing them, they can be used as a the basis
  vectors
  */

  /* calculate the axis */
  covMat.getEigenVectors(axis[0], axis[1], axis[2] );
  box.axis[0] = axis[0];
  box.axis[1] = axis[1];
  box.axis[2] = axis[2];

  PRINTF(3)("Eigenvectors:\n");
  PRINT(3)("\t\t%11.2f \t%11.2f \t%11.2f\n", box.axis[0].x, box.axis[0].y, box.axis[0].z);
  PRINT(3)("\t\t%11.2f \t%11.2f \t%11.2f\n", box.axis[1].x, box.axis[1].y, box.axis[1].z);
  PRINT(3)("\t\t%11.2f \t%11.2f \t%11.2f\n", box.axis[2].x, box.axis[2].y, box.axis[2].z);
}




/**
 *  calculate the eigenvectors for the object oriented box
 * @param box: reference to the box
 * @param modelInf: the model info structure of the model
 * @param tirangleIndexes: an array with the indexes of the triangles inside this
 * @param length: the length of the indexes array
 */
void OBBTreeNode::calculateBoxAxis(OBB& box, const modelInfo& modelInf, const int* triangleIndexes, int length)
{

  PRINTF(3)("Calculate Box Axis\n");
  /* now get the axis length */
  Line                ax[3];                                 //!< the axis
  float               halfLength[3];                         //!< half length of the axis
  float               tmpLength;                             //!< tmp save point for the length
  Plane               p0(box.axis[0], box.center);           //!< the axis planes
  Plane               p1(box.axis[1], box.center);           //!< the axis planes
  Plane               p2(box.axis[2], box.center);           //!< the axis planes
  float               maxLength[3];                          //!< maximal lenth of the axis
  float               minLength[3];                          //!< minimal length of the axis
  const sVec3D*       tmpVec;                                //!< variable taking tmp vectors


  /* get the maximal dimensions of the body in all directions */
  /* for the initialisation the value just has to be inside of the polygon soup -> first vertices (rand) */
  tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[0]].indexToVertices[0]]);
  maxLength[0] = p0.distancePoint(*tmpVec);
  minLength[0] = p0.distancePoint(*tmpVec);
  for( int j = 0; j < length; ++j)
  {
    for( int i = 0; i < 3; ++i)
    {
      tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[j]].indexToVertices[i]]);
      tmpLength = p0.distancePoint(*tmpVec);
      if( tmpLength > maxLength[0])
        maxLength[0] = tmpLength;
      else if( tmpLength < minLength[0])
        minLength[0] = tmpLength;
    }
  }

  /* for the initialisation the value just has to be inside of the polygon soup -> first vertices (rand) */
  tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[0]].indexToVertices[0]]);
  maxLength[1] = p1.distancePoint(*tmpVec);
  minLength[1] = p1.distancePoint(*tmpVec);
  for( int j = 0; j < length; ++j)
  {
    for( int i = 0; i < 3; ++i)
    {
      tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[j]].indexToVertices[i]]);
      tmpLength = p1.distancePoint(*tmpVec);
      if( tmpLength > maxLength[1])
        maxLength[1] = tmpLength;
      else if( tmpLength < minLength[1])
        minLength[1] = tmpLength;
    }
  }

  /* for the initialisation the value just has to be inside of the polygon soup -> first vertices (rand) */
  tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[0]].indexToVertices[0]]);
  maxLength[2] = p2.distancePoint(*tmpVec);
  minLength[2] = p2.distancePoint(*tmpVec);
  for( int j = 0; j < length; ++j)
  {
    for( int i = 0; i < 3; ++i)
    {
      tmpVec = (sVec3D*)(&modelInf.pVertices[modelInf.pTriangles[triangleIndexes[j]].indexToVertices[i]]);
      tmpLength = p2.distancePoint(*tmpVec);
      if( tmpLength > maxLength[2])
        maxLength[2] = tmpLength;
      else if( tmpLength < minLength[2])
        minLength[2] = tmpLength;
    }
  }


  /* calculate the real centre of the body by using the axis length */
  float               centerOffset[3];

  for( int i = 0; i < 3; ++i)
  {
    centerOffset[i] = (maxLength[i] + minLength[i]) / 2.0f;       // min length is negatie
    box.halfLength[i] = (maxLength[i]- minLength[i]) / 2.0f;      // min length is negative
  }
  box.center.x += centerOffset[0];
  box.center.y += centerOffset[1];
  box.center.z += centerOffset[2];

  PRINTF(3)("\n");
  PRINT(3)("\tAxis halflength x: %11.2f (max: %11.2f, \tmin: %11.2f)\n", box.halfLength[0], maxLength[0], minLength[0]);
  PRINT(3)("\tAxis halflength x: %11.2f (max: %11.2f, \tmin: %11.2f)\n", box.halfLength[1], maxLength[1], minLength[1]);
  PRINT(3)("\tAxis halflength x: %11.2f (max: %11.2f, \tmin: %11.2f)\n", box.halfLength[2], maxLength[2], minLength[2]);


//   box.halfLength[0] = halfLength[0];
//   box.halfLength[1] = halfLength[1];
//   box.halfLength[2] = halfLength[2];
}



/**
 *  this separates an ob-box in the middle
 * @param box: the box to separate
 *
 * this will separate the box into to smaller boxes. the separation is done along the middle of the longest axis
 */
void OBBTreeNode::forkBox(OBB& box)
{

  PRINTF(3)("Fork Box\n");
  PRINTF(4)("Calculating the longest Axis\n");
  /* get the longest axis of the box */
  float               longestAxis = -1.0f;                 //!< the length of the longest axis
  int                 longestAxisIndex = 0;                //!< this is the nr of the longest axis


  /* now get the longest axis of the three exiting */
  for( int i = 0; i < 3; ++i)
  {
    if( longestAxis < box.halfLength[i])
    {
      longestAxis = box.halfLength[i];
      longestAxisIndex = i;
    }
  }
  PRINTF(3)("\nLongest Axis is: Nr %i with a half-length of:%11.2f\n", longestAxisIndex, longestAxis);


  PRINTF(4)("Separating along the longest axis\n");
  /* get the closest vertex near the center */
  float               dist = 999999.0f;                    //!< the smallest distance to each vertex
  float               tmpDist;                             //!< variable to save diverse distances temporarily
  int                 vertexIndex;                         //!< index of the vertex near the center
  Plane               middlePlane(box.axis[longestAxisIndex], box.center); //!< the middle plane
  const sVec3D*       tmpVec;                              //!< temp simple 3D vector


  /* now definin the separation plane through this specified nearest point and partition
  the points depending on which side they are located
  */
  std::list<int>           partition1;                           //!< the vertex partition 1
  std::list<int>           partition2;                           //!< the vertex partition 2
  float*                   triangleCenter = new float[3];        //!< the center of the triangle
  const float*             a;                                    //!< triangle  edge a
  const float*             b;                                    //!< triangle  edge b
  const float*             c;                                    //!< triangle  edge c


  /* find the center of the box */
  this->separationPlane = Plane(box.axis[longestAxisIndex], box.center);
  this->sepPlaneCenter[0] = box.center.x;
  this->sepPlaneCenter[1] = box.center.y;
  this->sepPlaneCenter[2] = box.center.z;
  this->longestAxisIndex = longestAxisIndex;

  for( int i = 0; i < box.triangleIndexesLength; ++i)
  {
    /* first calculate the middle of the triangle */
    a = &box.modelInf->pVertices[box.modelInf->pTriangles[box.triangleIndexes[i]].indexToVertices[0]];
    b = &box.modelInf->pVertices[box.modelInf->pTriangles[box.triangleIndexes[i]].indexToVertices[1]];
    c = &box.modelInf->pVertices[box.modelInf->pTriangles[box.triangleIndexes[i]].indexToVertices[2]];

    triangleCenter[0] = (a[0] + b[0] + c[0]) / 3.0f;
    triangleCenter[1] = (a[1] + b[1] + c[1]) / 3.0f;
    triangleCenter[2] = (a[2] + b[2] + c[2]) / 3.0f;
    tmpDist = this->separationPlane.distancePoint(*((sVec3D*)triangleCenter));

    if( tmpDist > 0.0f)
      partition1.push_back(box.triangleIndexes[i]); /* positive numbers plus zero */
    else if( tmpDist < 0.0f)
      partition2.push_back(box.triangleIndexes[i]); /* negatice numbers */
    else {
      partition1.push_back(box.triangleIndexes[i]); /* 0.0f? unprobable... */
      partition2.push_back(box.triangleIndexes[i]);
    }
  }
  PRINTF(3)("\nPartition1: got \t%i Vertices \nPartition2: got \t%i Vertices\n", partition1.size(), partition2.size());


  /* now comes the separation into two different sVec3D arrays */
  int                index;                                //!< index storage place
  int*               triangleIndexList1;                   //!< the vertex list 1
  int*               triangleIndexList2;                   //!< the vertex list 2
  std::list<int>::iterator element;                        //!< the list iterator

  triangleIndexList1 = new int[partition1.size()];
  triangleIndexList2 = new int[partition2.size()];

  for( element = partition1.begin(), index = 0; element != partition1.end(); element++, index++)
    triangleIndexList1[index] = (*element);

  for( element = partition2.begin(), index = 0; element != partition2.end(); element++, index++)
    triangleIndexList2[index] = (*element);

  if( this->triangleIndexList1!= NULL)
    delete[] this->triangleIndexList1;
  this->triangleIndexList1 = triangleIndexList1;
  this->triangleIndexLength1 = partition1.size();

  if( this->triangleIndexList2 != NULL)
    delete[] this->triangleIndexList2;
  this->triangleIndexList2 = triangleIndexList2;
  this->triangleIndexLength2 = partition2.size();
}




void OBBTreeNode::collideWith(BVTreeNode* treeNode, WorldEntity* nodeA, WorldEntity* nodeB)
{
  if( unlikely(treeNode == NULL))
    return;

  PRINTF(3)("collideWith\n");
  /* if the obb overlap, make subtests: check which node is realy overlaping  */
  PRINTF(3)("Checking OBB %i vs %i: ", this->getIndex(), treeNode->getIndex());
  //   if( unlikely(treeNode == NULL)) return;


  if( this->overlapTest(*this->bvElement, *(((const OBBTreeNode*)&treeNode)->bvElement), nodeA, nodeB))
  {
    PRINTF(3)("collision @ lvl %i, object %s vs. %s, (%p, %p)\n", this->depth, nodeA->getClassName(), nodeB->getClassName(), this->nodeLeft, this->nodeRight);

    /* check if left node overlaps */
    if( likely( this->nodeLeft != NULL))
    {
      PRINTF(3)("Checking OBB %i vs %i: ", this->nodeLeft->getIndex(), treeNode->getIndex());
      if( this->overlapTest(*this->nodeLeft->bvElement, *(((const OBBTreeNode*)&treeNode)->bvElement), nodeA, nodeB))
      {
        this->nodeLeft->collideWith((((const OBBTreeNode*)treeNode)->nodeLeft), nodeA, nodeB);
        this->nodeLeft->collideWith((((const OBBTreeNode*)treeNode)->nodeRight), nodeA, nodeB);
      }
    }
    /* check if right node overlaps */
    if( likely( this->nodeRight != NULL))
    {
      PRINTF(3)("Checking OBB %i vs %i: ", this->nodeRight->getIndex(), treeNode->getIndex());
      if(this->overlapTest(*this->nodeRight->bvElement, *(((const OBBTreeNode*)&treeNode)->bvElement), nodeA, nodeB))
      {
        this->nodeRight->collideWith((((const OBBTreeNode*)treeNode)->nodeLeft), nodeA, nodeB);
        this->nodeRight->collideWith((((const OBBTreeNode*)treeNode)->nodeRight), nodeA, nodeB);
      }
    }

    /* so there is a collision and this is the last box in the tree (i.e. leaf) */
    /* FIXME: If we would choose || insead of && there would also be asymmetrical cases supported */
    if( unlikely(this->nodeRight == NULL && this->nodeLeft == NULL))
    {
      nodeA->collidesWith(nodeB, (((const OBBTreeNode*)&treeNode)->bvElement->center));

      nodeB->collidesWith(nodeA, this->bvElement->center);
    }

  }
}



bool OBBTreeNode::overlapTest(OBB& boxA, OBB& boxB, WorldEntity* nodeA, WorldEntity* nodeB)
{
  //HACK remove this again
  this->owner = nodeA;
  //   if( boxB == NULL || boxA == NULL)
  //     return false;

  /* first check all axis */
  Vector t;
  float rA = 0.0f;
  float rB = 0.0f;
  Vector l;
  Vector rotAxisA[3];
  Vector rotAxisB[3];

  rotAxisA[0] =  nodeA->getAbsDir().apply(boxA.axis[0]);
  rotAxisA[1] =  nodeA->getAbsDir().apply(boxA.axis[1]);
  rotAxisA[2] =  nodeA->getAbsDir().apply(boxA.axis[2]);

  rotAxisB[0] =  nodeB->getAbsDir().apply(boxB.axis[0]);
  rotAxisB[1] =  nodeB->getAbsDir().apply(boxB.axis[1]);
  rotAxisB[2] =  nodeB->getAbsDir().apply(boxB.axis[2]);


  t = nodeA->getAbsCoor() + nodeA->getAbsDir().apply(boxA.center) - ( nodeB->getAbsCoor() + nodeB->getAbsDir().apply(boxB.center));

  //   printf("\n");
  //   printf("(%f, %f, %f) -> (%f, %f, %f)\n", boxA->axis[0].x, boxA->axis[0].y, boxA->axis[0].z, rotAxisA[0].x, rotAxisA[0].y, rotAxisA[0].z);
  //   printf("(%f, %f, %f) -> (%f, %f, %f)\n", boxA->axis[1].x, boxA->axis[1].y, boxA->axis[1].z, rotAxisA[1].x, rotAxisA[1].y, rotAxisA[1].z);
  //   printf("(%f, %f, %f) -> (%f, %f, %f)\n", boxA->axis[2].x, boxA->axis[2].y, boxA->axis[2].z, rotAxisA[2].x, rotAxisA[2].y, rotAxisA[2].z);
  //
  //   printf("(%f, %f, %f) -> (%f, %f, %f)\n", boxB->axis[0].x, boxB->axis[0].y, boxB->axis[0].z, rotAxisB[0].x, rotAxisB[0].y, rotAxisB[0].z);
  //   printf("(%f, %f, %f) -> (%f, %f, %f)\n", boxB->axis[1].x, boxB->axis[1].y, boxB->axis[1].z, rotAxisB[1].x, rotAxisB[1].y, rotAxisB[1].z);
  //   printf("(%f, %f, %f) -> (%f, %f, %f)\n", boxB->axis[2].x, boxB->axis[2].y, boxB->axis[2].z, rotAxisB[2].x, rotAxisB[2].y, rotAxisB[2].z);


  /* All 3 axis of the object A */
  for( int j = 0; j < 3; ++j)
  {
    rA = 0.0f;
    rB = 0.0f;
    l = rotAxisA[j];

    rA += fabs(boxA.halfLength[0] * rotAxisA[0].dot(l));
    rA += fabs(boxA.halfLength[1] * rotAxisA[1].dot(l));
    rA += fabs(boxA.halfLength[2] * rotAxisA[2].dot(l));

    rB += fabs(boxB.halfLength[0] * rotAxisB[0].dot(l));
    rB += fabs(boxB.halfLength[1] * rotAxisB[1].dot(l));
    rB += fabs(boxB.halfLength[2] * rotAxisB[2].dot(l));

    PRINTF(3)("s = %f, rA+rB = %f\n", fabs(t.dot(l)), rA+rB);

    if( (rA + rB) < fabs(t.dot(l)))
    {
      PRINTF(3)("no Collision\n");
      return false;
    }
  }

  /* All 3 axis of the object B */
  for( int j = 0; j < 3; ++j)
  {
    rA = 0.0f;
    rB = 0.0f;
    l = rotAxisB[j];

    rA += fabs(boxA.halfLength[0] * rotAxisA[0].dot(l));
    rA += fabs(boxA.halfLength[1] * rotAxisA[1].dot(l));
    rA += fabs(boxA.halfLength[2] * rotAxisA[2].dot(l));

    rB += fabs(boxB.halfLength[0] * rotAxisB[0].dot(l));
    rB += fabs(boxB.halfLength[1] * rotAxisB[1].dot(l));
    rB += fabs(boxB.halfLength[2] * rotAxisB[2].dot(l));

    PRINTF(3)("s = %f, rA+rB = %f\n", fabs(t.dot(l)), rA+rB);

    if( (rA + rB) < fabs(t.dot(l)))
    {
      PRINTF(3)("no Collision\n");
      return false;
    }
  }


  /* Now check for all face cross products */

  for( int j = 0; j < 3; ++j)
  {
    for(int k = 0; k < 3; ++k )
    {
      rA = 0.0f;
      rB = 0.0f;
      l = rotAxisA[j].cross(rotAxisB[k]);

      rA += fabs(boxA.halfLength[0] * rotAxisA[0].dot(l));
      rA += fabs(boxA.halfLength[1] * rotAxisA[1].dot(l));
      rA += fabs(boxA.halfLength[2] * rotAxisA[2].dot(l));

      rB += fabs(boxB.halfLength[0] * rotAxisB[0].dot(l));
      rB += fabs(boxB.halfLength[1] * rotAxisB[1].dot(l));
      rB += fabs(boxB.halfLength[2] * rotAxisB[2].dot(l));

      PRINTF(3)("s = %f, rA+rB = %f\n", fabs(t.dot(l)), rA+rB);

      if( (rA + rB) < fabs(t.dot(l)))
      {
        PRINTF(3)("keine Kollision\n");
        return false;
      }
    }
  }

  /* FIXME: there is no collision mark set now */
     boxA.bCollided = true; /* use this ONLY(!!!!) for drawing operations */
     boxB.bCollided = true;


  PRINTF(3)("Kollision!\n");
  return true;
}










/**
 *
 * draw the BV tree - debug mode
 */
void OBBTreeNode::drawBV(int depth, int drawMode, const Vector& color,  bool top) const
{
  /* this function can be used to draw the triangles and/or the points only  */
  if( drawMode & DRAW_MODEL || drawMode & DRAW_ALL)
  {
    if( !(drawMode & DRAW_SINGLE && depth != 0))
    {
      if( drawMode & DRAW_POINTS)
      {
        glBegin(GL_POINTS);
        for( int i = 0; i < this->bvElement->modelInf->numVertices*3; i+=3)
          glVertex3f(this->bvElement->modelInf->pVertices[i],
                     this->bvElement->modelInf->pVertices[i+1],
                     this->bvElement->modelInf->pVertices[i+2]);
        glEnd();
      }
    }
  }

  if (top)
  {
    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_LIGHTING);
    glDisable(GL_TEXTURE_2D);
  }
  glColor3f(color.x, color.y, color.z);


  /* draw world axes */
  if( drawMode & DRAW_BV_AXIS)
  {
    glBegin(GL_LINES);
    glColor3f(1.0, 0.0, 0.0);
    glVertex3f(0.0, 0.0, 0.0);
    glVertex3f(3.0, 0.0, 0.0);

    glColor3f(0.0, 1.0, 0.0);
    glVertex3f(0.0, 0.0, 0.0);
    glVertex3f(0.0, 3.0, 0.0);

    glColor3f(0.0, 0.0, 1.0);
    glVertex3f(0.0, 0.0, 0.0);
    glVertex3f(0.0, 0.0, 3.0);
    glEnd();
  }


  if( drawMode & DRAW_BV_AXIS || drawMode & DRAW_ALL)
  {
    if( drawMode & DRAW_SINGLE && depth != 0)
    {
      /* draw the obb axes */
      glBegin(GL_LINES);
      glColor3f(1.0, 0.0, 0.0);
      glVertex3f(this->bvElement->center.x, this->bvElement->center.y, this->bvElement->center.z);
      glVertex3f(this->bvElement->center.x + this->bvElement->axis[0].x * this->bvElement->halfLength[0],
                 this->bvElement->center.y + this->bvElement->axis[0].y * this->bvElement->halfLength[0],
                 this->bvElement->center.z + this->bvElement->axis[0].z * this->bvElement->halfLength[0]);

      glColor3f(0.0, 1.0, 0.0);
      glVertex3f(this->bvElement->center.x, this->bvElement->center.y, this->bvElement->center.z);
      glVertex3f(this->bvElement->center.x + this->bvElement->axis[1].x * this->bvElement->halfLength[1],
                 this->bvElement->center.y + this->bvElement->axis[1].y * this->bvElement->halfLength[1],
                 this->bvElement->center.z + this->bvElement->axis[1].z * this->bvElement->halfLength[1]);

      glColor3f(0.0, 0.0, 1.0);
      glVertex3f(this->bvElement->center.x, this->bvElement->center.y, this->bvElement->center.z);
      glVertex3f(this->bvElement->center.x + this->bvElement->axis[2].x * this->bvElement->halfLength[2],
                 this->bvElement->center.y + this->bvElement->axis[2].y * this->bvElement->halfLength[2],
                 this->bvElement->center.z + this->bvElement->axis[2].z * this->bvElement->halfLength[2]);
      glEnd();
    }
  }


  /* DRAW POLYGONS */
  if( drawMode & DRAW_BV_POLYGON || drawMode & DRAW_ALL || drawMode & DRAW_BV_BLENDED)
  {
    if (top)
    {
      glEnable(GL_BLEND);
      glBlendFunc(GL_SRC_ALPHA, GL_ONE);
    }

    if( this->nodeLeft == NULL && this->nodeRight == NULL)
      depth = 0;

    if( depth == 0 /*!(drawMode & DRAW_SINGLE && depth != 0)*/)
    {


      Vector cen = this->bvElement->center;
      Vector* axis = this->bvElement->axis;
      float* len = this->bvElement->halfLength;

      if( this->bvElement->bCollided)
      {
        glColor4f(1.0, 1.0, 1.0, .5); // COLLISION COLOR
      }
      else if( drawMode & DRAW_BV_BLENDED)
      {
        glColor4f(color.x, color.y, color.z, .5);
      }

      // debug out
      if( this->obbTree->getOwner() != NULL)
      {
        PRINTF(4)("debug poly draw: depth: %i, mode: %i, entity-name: %s, class: %s\n", depth, drawMode, this->obbTree->getOwner()->getName(), this->obbTree->getOwner()->getClassName());
      }
      else
        PRINTF(4)("debug poly draw: depth: %i, mode: %i\n", depth, drawMode);


      /* draw bounding box */
      if( drawMode & DRAW_BV_BLENDED)
        glBegin(GL_QUADS);
      else
        glBegin(GL_LINE_LOOP);
      glVertex3f(cen.x + axis[0].x * len[0] + axis[1].x * len[1] + axis[2].x * len[2],
                 cen.y + axis[0].y * len[0] + axis[1].y * len[1] + axis[2].y * len[2],
                 cen.z + axis[0].z * len[0] + axis[1].z * len[1] + axis[2].z * len[2]);
      glVertex3f(cen.x + axis[0].x * len[0] + axis[1].x * len[1] - axis[2].x * len[2],
                 cen.y + axis[0].y * len[0] + axis[1].y * len[1] - axis[2].y * len[2],
                 cen.z + axis[0].z * len[0] + axis[1].z * len[1] - axis[2].z * len[2]);
      glVertex3f(cen.x + axis[0].x * len[0] - axis[1].x * len[1] - axis[2].x * len[2],
                 cen.y + axis[0].y * len[0] - axis[1].y * len[1] - axis[2].y * len[2],
                 cen.z + axis[0].z * len[0] - axis[1].z * len[1] - axis[2].z * len[2]);
      glVertex3f(cen.x + axis[0].x * len[0] - axis[1].x * len[1] + axis[2].x * len[2],
                 cen.y + axis[0].y * len[0] - axis[1].y * len[1] + axis[2].y * len[2],
                 cen.z + axis[0].z * len[0] - axis[1].z * len[1] + axis[2].z * len[2]);
      glEnd();

      if( drawMode & DRAW_BV_BLENDED)
        glBegin(GL_QUADS);
      else
        glBegin(GL_LINE_LOOP);
      glVertex3f(cen.x + axis[0].x * len[0] - axis[1].x * len[1] + axis[2].x * len[2],
                 cen.y + axis[0].y * len[0] - axis[1].y * len[1] + axis[2].y * len[2],
                 cen.z + axis[0].z * len[0] - axis[1].z * len[1] + axis[2].z * len[2]);
      glVertex3f(cen.x + axis[0].x * len[0] - axis[1].x * len[1] - axis[2].x * len[2],
                 cen.y + axis[0].y * len[0] - axis[1].y * len[1] - axis[2].y * len[2],
                 cen.z + axis[0].z * len[0] - axis[1].z * len[1] - axis[2].z * len[2]);
      glVertex3f(cen.x - axis[0].x * len[0] - axis[1].x * len[1] - axis[2].x * len[2],
                 cen.y - axis[0].y * len[0] - axis[1].y * len[1] - axis[2].y * len[2],
                 cen.z - axis[0].z * len[0] - axis[1].z * len[1] - axis[2].z * len[2]);
      glVertex3f(cen.x - axis[0].x * len[0] - axis[1].x * len[1] + axis[2].x * len[2],
                 cen.y - axis[0].y * len[0] - axis[1].y * len[1] + axis[2].y * len[2],
                 cen.z - axis[0].z * len[0] - axis[1].z * len[1] + axis[2].z * len[2]);
      glEnd();

      if( drawMode & DRAW_BV_BLENDED)
        glBegin(GL_QUADS);
      else
        glBegin(GL_LINE_LOOP);
      glVertex3f(cen.x - axis[0].x * len[0] - axis[1].x * len[1] + axis[2].x * len[2],
                 cen.y - axis[0].y * len[0] - axis[1].y * len[1] + axis[2].y * len[2],
                 cen.z - axis[0].z * len[0] - axis[1].z * len[1] + axis[2].z * len[2]);
      glVertex3f(cen.x - axis[0].x * len[0] - axis[1].x * len[1] - axis[2].x * len[2],
                 cen.y - axis[0].y * len[0] - axis[1].y * len[1] - axis[2].y * len[2],
                 cen.z - axis[0].z * len[0] - axis[1].z * len[1] - axis[2].z * len[2]);
      glVertex3f(cen.x - axis[0].x * len[0] + axis[1].x * len[1] - axis[2].x * len[2],
                 cen.y - axis[0].y * len[0] + axis[1].y * len[1] - axis[2].y * len[2],
                 cen.z - axis[0].z * len[0] + axis[1].z * len[1] - axis[2].z * len[2]);
      glVertex3f(cen.x - axis[0].x * len[0] + axis[1].x * len[1] + axis[2].x * len[2],
                 cen.y - axis[0].y * len[0] + axis[1].y * len[1] + axis[2].y * len[2],
                 cen.z - axis[0].z * len[0] + axis[1].z * len[1] + axis[2].z * len[2]);
      glEnd();

      if( drawMode & DRAW_BV_BLENDED)
        glBegin(GL_QUADS);
      else
        glBegin(GL_LINE_LOOP);
      glVertex3f(cen.x - axis[0].x * len[0] + axis[1].x * len[1] - axis[2].x * len[2],
                 cen.y - axis[0].y * len[0] + axis[1].y * len[1] - axis[2].y * len[2],
                 cen.z - axis[0].z * len[0] + axis[1].z * len[1] - axis[2].z * len[2]);
      glVertex3f(cen.x - axis[0].x * len[0] + axis[1].x * len[1] + axis[2].x * len[2],
                 cen.y - axis[0].y * len[0] + axis[1].y * len[1] + axis[2].y * len[2],
                 cen.z - axis[0].z * len[0] + axis[1].z * len[1] + axis[2].z * len[2]);
      glVertex3f(cen.x + axis[0].x * len[0] + axis[1].x * len[1] + axis[2].x * len[2],
                 cen.y + axis[0].y * len[0] + axis[1].y * len[1] + axis[2].y * len[2],
                 cen.z + axis[0].z * len[0] + axis[1].z * len[1] + axis[2].z * len[2]);
      glVertex3f(cen.x + axis[0].x * len[0] + axis[1].x * len[1] - axis[2].x * len[2],
                 cen.y + axis[0].y * len[0] + axis[1].y * len[1] - axis[2].y * len[2],
                 cen.z + axis[0].z * len[0] + axis[1].z * len[1] - axis[2].z * len[2]);
      glEnd();


      if( drawMode & DRAW_BV_BLENDED)
      {
        glBegin(GL_QUADS);
        glVertex3f(cen.x - axis[0].x * len[0] + axis[1].x * len[1] - axis[2].x * len[2],
                   cen.y - axis[0].y * len[0] + axis[1].y * len[1] - axis[2].y * len[2],
                   cen.z - axis[0].z * len[0] + axis[1].z * len[1] - axis[2].z * len[2]);
        glVertex3f(cen.x + axis[0].x * len[0] + axis[1].x * len[1] - axis[2].x * len[2],
                   cen.y + axis[0].y * len[0] + axis[1].y * len[1] - axis[2].y * len[2],
                   cen.z + axis[0].z * len[0] + axis[1].z * len[1] - axis[2].z * len[2]);
        glVertex3f(cen.x + axis[0].x * len[0] - axis[1].x * len[1] - axis[2].x * len[2],
                   cen.y + axis[0].y * len[0] - axis[1].y * len[1] - axis[2].y * len[2],
                   cen.z + axis[0].z * len[0] - axis[1].z * len[1] - axis[2].z * len[2]);
        glVertex3f(cen.x - axis[0].x * len[0] - axis[1].x * len[1] - axis[2].x * len[2],
                   cen.y - axis[0].y * len[0] - axis[1].y * len[1] - axis[2].y * len[2],
                   cen.z - axis[0].z * len[0] - axis[1].z * len[1] - axis[2].z * len[2]);
        glEnd();

        glBegin(GL_QUADS);
        glVertex3f(cen.x - axis[0].x * len[0] + axis[1].x * len[1] + axis[2].x * len[2],
                   cen.y - axis[0].y * len[0] + axis[1].y * len[1] + axis[2].y * len[2],
                   cen.z - axis[0].z * len[0] + axis[1].z * len[1] + axis[2].z * len[2]);
        glVertex3f(cen.x + axis[0].x * len[0] + axis[1].x * len[1] + axis[2].x * len[2],
                   cen.y + axis[0].y * len[0] + axis[1].y * len[1] + axis[2].y * len[2],
                   cen.z + axis[0].z * len[0] + axis[1].z * len[1] + axis[2].z * len[2]);
        glVertex3f(cen.x + axis[0].x * len[0] - axis[1].x * len[1] + axis[2].x * len[2],
                   cen.y + axis[0].y * len[0] - axis[1].y * len[1] + axis[2].y * len[2],
                   cen.z + axis[0].z * len[0] - axis[1].z * len[1] + axis[2].z * len[2]);
        glVertex3f(cen.x - axis[0].x * len[0] - axis[1].x * len[1] + axis[2].x * len[2],
                   cen.y - axis[0].y * len[0] - axis[1].y * len[1] + axis[2].y * len[2],
                   cen.z - axis[0].z * len[0] - axis[1].z * len[1] + axis[2].z * len[2]);
        glEnd();
      }

      if( drawMode & DRAW_BV_BLENDED)
        glColor3f(color.x, color.y, color.z);
    }
  }

  /* DRAW SEPARATING PLANE */
  if( drawMode & DRAW_SEPARATING_PLANE || drawMode & DRAW_ALL)
  {
    if( !(drawMode & DRAW_SINGLE && depth != 0))
    {
      if( drawMode & DRAW_BV_BLENDED)
        glColor4f(color.x, color.y, color.z, .6);

      /* now draw the separation plane */
      Vector a1 = this->bvElement->axis[(this->longestAxisIndex + 1)%3];
      Vector a2 = this->bvElement->axis[(this->longestAxisIndex + 2)%3];
      Vector c = this->bvElement->center;
      float l1 = this->bvElement->halfLength[(this->longestAxisIndex + 1)%3];
      float l2 = this->bvElement->halfLength[(this->longestAxisIndex + 2)%3];
      glBegin(GL_QUADS);
      glVertex3f(c.x + a1.x * l1 + a2.x * l2, c.y + a1.y * l1+ a2.y * l2, c.z + a1.z * l1 + a2.z * l2);
      glVertex3f(c.x - a1.x * l1 + a2.x * l2, c.y - a1.y * l1+ a2.y * l2, c.z - a1.z * l1 + a2.z * l2);
      glVertex3f(c.x - a1.x * l1 - a2.x * l2, c.y - a1.y * l1- a2.y * l2, c.z - a1.z * l1 - a2.z * l2);
      glVertex3f(c.x + a1.x * l1 - a2.x * l2, c.y + a1.y * l1- a2.y * l2, c.z + a1.z * l1 - a2.z * l2);
      glEnd();

      if( drawMode & DRAW_BV_BLENDED)
        glColor4f(color.x, color.y, color.z, 1.0);

    }
  }



  if (depth > 0)
  {
    if( this->nodeLeft != NULL)
      this->nodeLeft->drawBV(depth - 1, drawMode, Color::HSVtoRGB(Color::RGBtoHSV(color)+Vector(15.0,0.0,0.0)), false);
    if( this->nodeRight != NULL)
      this->nodeRight->drawBV(depth - 1, drawMode, Color::HSVtoRGB(Color::RGBtoHSV(color)+Vector(30.0,0.0,0.0)), false);
  }
  this->bvElement->bCollided = false;

  if (top)
    glPopAttrib();
}



void OBBTreeNode::debug() const
{
  PRINT(0)("========OBBTreeNode::debug()=====\n");
  PRINT(0)(" Current depth: %i", this->depth);
  PRINT(0)(" ");
  PRINT(0)("=================================\n");
}
