/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "material.h"

#include "texture.h"
#include "debug.h"
#include "util/loading/resource_manager.h"
#include <stdlib.h>
#include <string.h>

//! @todo check if we are in RESOURCE MANAGER-mode
#include "util/loading/resource_manager.h"

using namespace std;

/**
 * creates a Material.
 * @param mtlName Name of the Material to be added to the Material List
 */
Material::Material (const std::string& mtlName)
{
  this->setClassID(CL_MATERIAL, "Material");

  this->setIllum(3);
  this->setDiffuse(1,1,1);
  this->setAmbient(0,0,0);
  this->setSpecular(.5,.5,.5);
  this->setShininess(2.0);
  this->setTransparency(1.0);

  this->ambientTexture = NULL;
  this->specularTexture = NULL;
  this->sFactor = GL_SRC_ALPHA;
  this->tFactor = GL_ONE;

  this->setName(mtlName);
}

/**
  *  deletes a Material
*/
Material::~Material()
{
  PRINTF(5)("delete Material %s.\n", this->getName());

  while(!this->textures.empty())
  {
    if (this->textures.back() != NULL)
      ResourceManager::getInstance()->unload(this->textures.back());
    this->textures.pop_back();
  }

  if (this->ambientTexture != NULL)
    ResourceManager::getInstance()->unload(this->ambientTexture);
  if (this->specularTexture != NULL)
    ResourceManager::getInstance()->unload(this->specularTexture);

  if (this == Material::selectedMaterial)
    Material::selectedMaterial = NULL;
}


Material* Material::selectedMaterial = NULL;

const GLenum Material::glTextureArbs[] =
{
  GL_TEXTURE0_ARB,
  GL_TEXTURE1_ARB,
  GL_TEXTURE2_ARB,
  GL_TEXTURE3_ARB,
  GL_TEXTURE4_ARB,
  GL_TEXTURE5_ARB,
  GL_TEXTURE6_ARB,
  GL_TEXTURE7_ARB
};


/// TODO FIX THIS
// Material& Material::operator=(const Material& m)
// {
//   this->setIllum(m.illumModel);
//   this->setDiffuse(m.diffuse[0],m.diffuse[1],m.diffuse[2]);
//   this->setAmbient(m.ambient[0],m.ambient[1],m.ambient[2]);
//   this->setSpecular(m.specular[0],m.specular[1],m.specular[2]);
//   this->setShininess(m.shininess);
//   this->setTransparency(m.transparency);
//
//   if (this->diffuseTexture != NULL)
//     ResourceManager::getInstance()->unload(this->diffuseTexture);
//   if (m.diffuseTexture != NULL)
//     this->diffuseTexture = (Texture*)ResourceManager::getInstance()->copy(m.diffuseTexture);
//   this->ambientTexture = NULL; /// FIXME
//   this->specularTexture = NULL; /// FIXME
//
//   this->setName(m.getName());
// }



/**
 *  sets the material with which the following Faces will be painted
 */
bool Material::select() const
{
  if (unlikely(this == Material::selectedMaterial))
      return true;


  // setting diffuse color
  glColor4f (diffuse[0], diffuse[1], diffuse[2], this->transparency);
  // setting ambient color
  glMaterialfv(GL_FRONT, GL_AMBIENT, this->ambient);
  // setting up Sprecular
  glMaterialfv(GL_FRONT, GL_SPECULAR, this->specular);
  // setting up Shininess
  glMaterialf(GL_FRONT, GL_SHININESS, this->shininess);


  // setting the transparency
  if (this->transparency < 1.0 ||       /* This allows alpha blending of 2D textures with the scene */
      (likely(!this->textures.empty() && this->textures[0] != NULL) && this->textures[0]->hasAlpha()))
  {
    glEnable(GL_BLEND);
    glBlendFunc(this->sFactor, this->tFactor);
  }
  else
  {
    glDisable(GL_BLEND);
  }


  // setting illumination Model
  if (this->illumModel == 1) //! @todo make this work, if no vertex-normals are read.
    glShadeModel(GL_FLAT);
  else if (this->illumModel >= 2)
    glShadeModel(GL_SMOOTH);

  if (likely(Material::selectedMaterial != NULL))
  {
    for(unsigned int i = 0; i < Material::selectedMaterial->textures.size(); ++i)
    {
        glActiveTexture(Material::glTextureArbs[i]);
        glBindTexture(GL_TEXTURE_2D, 0);
    }
  }

  for(unsigned int i = 0; i < this->textures.size(); ++i)
  {
    if (likely(this->textures[i] != NULL))
    {
      glActiveTexture(Material::glTextureArbs[i]);
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, this->textures[i]->getTexture());
    }
  }

  /*  if (this->diffuseTexture != NULL)
      {
        glEnable(GL_TEXTURE_2D);
        glBindTexture(GL_TEXTURE_2D, this->diffuseTexture->getTexture());
      }
    else
      {
        glDisable(GL_TEXTURE_2D);
        glBindTexture(GL_TEXTURE_2D, 0);
      }*/
}

/**
 *  Sets the Material Illumination Model.
 *  illu illumination Model in int form
 */
void Material::setIllum (int illum)
{
  PRINTF(4)("setting illumModel of Material %s to %i\n", this->getName(), illum);
  this->illumModel = illum;
}

/**
 *  Sets the Material Illumination Model.
 *  illu illumination Model in char* form
 */
void Material::setIllum (char* illum)
{
  this->setIllum (atoi(illum));
}

/**
 *  Sets the Material Diffuse Color.
 * @param r Red Color Channel.a
 * @param g Green Color Channel.
 * @param b Blue Color Channel.
 */
void Material::setDiffuse (float r, float g, float b)
{
  PRINTF(4)("setting Diffuse Color of Material %s to r=%f g=%f b=%f.\n", this->getName(), r, g, b);
  this->diffuse[0] = r;
  this->diffuse[1] = g;
  this->diffuse[2] = b;
  this->diffuse[3] = 1.0;

}

/**
 *  Sets the Material Diffuse Color.
 * @param rgb The red, green, blue channel in char format (with spaces between them)
 */
void Material::setDiffuse (char* rgb)
{
  float r,g,b;
  sscanf (rgb, "%f %f %f", &r, &g, &b);
  this->setDiffuse (r, g, b);
}

/**
 *  Sets the Material Ambient Color.
 * @param r Red Color Channel.
 * @param g Green Color Channel.
 * @param b Blue Color Channel.
*/
void Material::setAmbient (float r, float g, float b)
{
  PRINTF(4)("setting Ambient Color of Material %s to r=%f g=%f b=%f.\n", this->getName(), r, g, b);
  this->ambient[0] = r;
  this->ambient[1] = g;
  this->ambient[2] = b;
  this->ambient[3] = 1.0;
}

/**
 *  Sets the Material Ambient Color.
 * @param rgb The red, green, blue channel in char format (with spaces between them)
 */
void Material::setAmbient (char* rgb)
{
  float r,g,b;
  sscanf (rgb, "%f %f %f", &r, &g, &b);
  this->setAmbient (r, g, b);
}

/**
 *  Sets the Material Specular Color.
 * @param r Red Color Channel.
 * @param g Green Color Channel.
 * @param b Blue Color Channel.
 */
void Material::setSpecular (float r, float g, float b)
{
  PRINTF(4)("setting Specular Color of Material %s to r=%f g=%f b=%f.\n", this->getName(), r, g, b);
  this->specular[0] = r;
  this->specular[1] = g;
  this->specular[2] = b;
  this->specular[3] = 1.0;
}

/**
 *  Sets the Material Specular Color.
 * @param rgb The red, green, blue channel in char format (with spaces between them)
*/
void Material::setSpecular (char* rgb)
{
  float r,g,b;
  sscanf (rgb, "%f %f %f", &r, &g, &b);
  this->setSpecular (r, g, b);
}

/**
 *  Sets the Material Shininess.
 * @param shini stes the Shininess from float.
*/
void Material::setShininess (float shini)
{
  this->shininess = shini;
}
/**
 *  Sets the Material Shininess.
 * @param shini stes the Shininess from char*.
*/
void Material::setShininess (char* shini)
{
  this->setShininess (atof(shini));
}

/**
 *  Sets the Material Transparency.
 * @param trans stes the Transparency from int.
*/
void Material::setTransparency (float trans)
{
  PRINTF(4)("setting Transparency of Material %s to %f.\n", this->getName(), trans);
  this->transparency = trans;
}
/**
 *  Sets the Material Transparency.
 * @param trans stes the Transparency from char*.
*/
void Material::setTransparency (char* trans)
{
  this->setTransparency (atof(trans));
}

/**
 *  Adds a Texture Path to the List of already existing Paths
 * @param pathName The Path to add.
*/
void Material::addTexturePath(const std::string& pathName)
{
  ResourceManager::getInstance()->addImageDir(pathName);
}

// MAPPING //

/**
 *  Sets the Materials Diffuse Map
 * @param dMap the Name of the Image to Use
*/
void Material::setDiffuseMap(const std::string& dMap, GLenum target, unsigned int textureNumber)
{
  assert(textureNumber < Material::getMaxTextureUnits());

  PRINTF(5)("setting Diffuse Map %s\n", dMap);
  if (this->textures.size() > textureNumber && this->textures[textureNumber] != NULL)
    ResourceManager::getInstance()->unload(this->textures[textureNumber]);

  if (this->textures.size() <= textureNumber)
    this->textures.resize(textureNumber+1, NULL);

  //! @todo check if RESOURCE MANAGER is availiable
  if (!dMap.empty())
  {

    this->textures[textureNumber] = (Texture*)ResourceManager::getInstance()->load(dMap, IMAGE, RP_GAME, (int)target);
  }
  else
  {
    this->textures[textureNumber] = NULL;
  }
}

/**
 *  Sets the Materials Ambient Map
 * @param aMap the Name of the Image to Use
   @todo implement this
*/
void Material::setAmbientMap(const std::string& aMap, GLenum target)
{
  SDL_Surface* ambientMap;

}

/**
 *  Sets the Materials Specular Map
 * @param sMap the Name of the Image to Use
   @todo implement this
*/
void Material::setSpecularMap(const std::string& sMap, GLenum target)
{
  SDL_Surface* specularMap;

}

/**
 *  Sets the Materials Bumpiness
 * @param bump the Name of the Image to Use
   @todo implemet this
*/
void Material::setBump(const std::string& bump)
{
}



int Material::getMaxTextureUnits()
{
  int maxTexUnits = 0;
  glGetIntegerv(GL_MAX_TEXTURE_UNITS, &maxTexUnits);
  return maxTexUnits;
}
