/*!
 * @file synchronizeable_var.h
 * @brief Definition of SynchronizeableVar
*/

#ifndef _SYNCHRONIZEABLE_VAR_H
#define _SYNCHRONIZEABLE_VAR_H

#include <string>
#include "netdefs.h"

class SynchronizeableVar {

  public:
    SynchronizeableVar( void * ptrIn, void * ptrOut, std::string name, int length, int permission = 0, int priority = 0 );
    virtual ~SynchronizeableVar();
    
    /**
     * check if synchronizeable wants to be informed on changes
     * @return true if synchronizeable wants to be informed on changes
     */
    inline bool beWatched(){ return this->bWatched; }
    
    /**
     * write var data to byte buffer
     * @param buf pointer to write to
     * @param maxLength writeToBuf will not write more than maxLength bytes
     * @return number bytes written
     */
    virtual int writeToBuf( byte * buf, int maxLength ) = 0;
    
    /**
     * read var data from byte buffer
     * @param buf pointer to read from
     * @param maxLength readFromBuf will not read more than maxLength bytes
     * @return number bytes read
     */
    virtual int readFromBuf( byte * buf, int maxLength ) = 0;
    
    /**
     * check if writeToBuf will return the same size every time
     * @return true if same size every time
     */
    virtual bool hasStaticSize() = 0;
    
    /**
     * get size writeToBuf needs
     * @return size in bytes
     */
    virtual int getSize(){ return length; }
    
    /**
     * check for permission to write
     * @return true if you can write
     */
    inline bool checkPremission( int permission ){ return (permission & this->permission) != 0; }
    
    /**
     * get variable name
     * @return name
     */
    inline std::string getName(){ return name; }
    
    /**
     * set variable name
     * @param name new name
     */
    inline void setName( std::string name ) { this->name = name; }
    
    /**
     * get priority
     * @return priority
     */
    inline int getPriority() { return this->priority; }
    
    /**
     * set priority
     * @param p priority
     */
    inline void setPriority( int p ) { this->priority = p; }
    
    /**
     * reset priority to variable specific default value
     */
    inline void resetPriority() { this->priority = this->real_priority; }


  private:
    bool bWatched;      //!< true if synchronizeable wants to be informed on changes

  protected:
    void * ptrIn;       //!< pointer to data (read)
    void * ptrOut;      //!< pointer to data (write)
    int length;         //!< data length
    int permission;     //!< who is allowed to change this var
    int priority;       //!< priority assigned to var
    int real_priority;  //!< priority assigned to var, increased every time not sent
    
    
    std::string name;    //!< variable name (for debugging)

};

#endif /* _PROTO_CLASS_H */
