/*!
 * @file text.h
 * @brief Definition of a text Class, that is able to render text.
 */

#ifndef _TEXT_H
#define _TEXT_H

#include "element_2d.h"

#define  TEXT_ALIGN_LEFT             E2D_ALIGN_LEFT
#define  TEXT_ALIGN_RIGHT            E2D_ALIGN_RIGHT
#define  TEXT_ALIGN_CENTER           E2D_ALIGN_CENTER
#define  TEXT_ALIGN_SCREEN_CENTER    E2D_ALIGN_SCREEN_CENTER
#define  TEXT_DEFAULT_COLOR          Vector(1.0, 1.0, 1.0)      //!< the default Color (white)
#define  TEXT_DEFAULT_BLENDING       1.0f                       //!< the default blending of the text, (no blending at all)

#define  TEXT_DEFAULT_ALIGNMENT      TEXT_ALIGN_LEFT            //!< default alignment
#define  TEXT_DEFAULT_SIZE           20                         //!< default size of the Text

// FORWARD DECLARATION
class Font;
struct SDL_Surface;

//! Represents one textElement.
class Text : public Element2D
{
  public:
    Text(const std::string& fontFile = "", unsigned int fontSize = TEXT_DEFAULT_SIZE);
    virtual ~Text();

    // SETUP
    void setFont(const std::string& fontFile, unsigned int renderSize);
    void setText(const std::string& text);
    /** @param blending the blending intensity to set (between 0.0 and 1.0) */
    inline void setBlending(float blending) { this->blending = blending; };
    /** @param r red @param g green @param b blue @brief sets the Color of the Text to render (values in [0-1]) */
    void setColor(float r, float g, float b) { this->color = Vector(r, g, b); };
    /** @param size the Size of the Text @brief sets the Size of the Font */
    inline void setSize(float size) { this->setSizeY2D(size); };

    /// RETRIEVE
    /** @returns the pointer to the stored Font (not changeable) */
    inline const Font* const getFont() const { return this->font; };
    /** @returns the String this Text displays */
    inline const std::string& getText() const { return this->text; };
    /** @returns the Blending Value [0 invisible 1.0 full visible */
    inline float getBlending() const { return this->blending; };
    /** @returns: a Vector(r,g,b) @brief: retrieve a Vector holding the Color of the Text */
    inline const Vector& getColor() const { return this->color; };
    /** @returns the Size of the Text */
    inline float getSize() const { return this->getSizeY2D(); };

    virtual void draw() const;

    void debug() const;

  protected:
    virtual void setupTextWidth();
  private:
    Font*             font;           //!< Font of this text

    std::string       text;           //!< The text to display
    Vector            color;          //!< The color of the font.
    float             blending;       //!< The blending intensity.
};

#endif /* _TEXT_H */
