/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_GRAPHICS

#include "text.h"
#include "font.h"

#include "util/loading/resource_manager.h"
#include "debug.h"

using namespace std;

/**
 * @brief creates a new Text Element
 * @param fontFile the Font to render this text in
 * @param type The renderType to display this font in
 */
Text::Text(const std::string& fontFile, unsigned int textSize)
{
  this->setClassID(CL_TEXT, "Text");

  // initialize this Text
  this->font = NULL;
  this->text = "";
  this->setAlignment(TEXT_DEFAULT_ALIGNMENT);
  this->blending = TEXT_DEFAULT_BLENDING;
  this->color = TEXT_DEFAULT_COLOR;
  this->setSize(TEXT_DEFAULT_SIZE);
  this->setText("");


  if (!fontFile.empty())
    this->setFont(fontFile, FONT_DEFAULT_RENDER_SIZE);
  this->setSizeY2D(textSize);
}

/**
 * @brief deletes a Text out of memory
 */
Text::~Text()
{
  if (this->font != NULL && this->font != Font::getDefaultFont())
    ResourceManager::getInstance()->unload(this->font);
}


/**
 * @brief sets the Font of this Text to font from fontFile
 * @param fontFile the File to load the Font from.
 * @param fontSize the Size of the Font
 */
void Text::setFont(const std::string& fontFile, unsigned int fontSize)
{
  Font* newFont;
  Font* oldFont = this->font;

  // load a new Font
  if (!fontFile.empty())
  {
    newFont = (Font*)ResourceManager::getInstance()->load(fontFile, TTF, RP_GAME, (int)fontSize);
    if (newFont == NULL)
    {
      newFont = Font::getDefaultFont();
      PRINTF(2)("Font %s could not be loaded, probably file not found\n", fontFile.c_str());
    }
  }
  else
    newFont = Font::getDefaultFont();

  // unloading the Font if we alrady have one loaded.
  this->font = newFont;
  if (oldFont != NULL && oldFont != Font::getDefaultFont())
    ResourceManager::getInstance()->unload(oldFont);

  this->setupTextWidth();
}

/**
 * @brief Sets a new Text to the font
 * @param text the new text to set
 */
void Text::setText(const std::string& text)
{
  this->text = text;

  this->setupTextWidth();
}

/**
 * @brief draws the Text
 */
void Text::draw() const
{
  if (unlikely(this->text.empty()))
    return;
  glPushMatrix();
  // transform for alignment.
  if (this->getAlignment() == TEXT_ALIGN_RIGHT)
    glTranslatef(-this->getSizeX2D(), 0, 0);
  else if (this->getAlignment() == TEXT_ALIGN_CENTER || this->getAlignment() == TEXT_ALIGN_SCREEN_CENTER)
    glTranslatef(-this->getSizeX2D()/2, 0, 0);

  // drawing this Text.
  // setting the Blending effects
  glColor4f(this->color.x, this->color.y, this->color.z, this->blending);
  glEnable(GL_BLEND);
  glEnable(GL_TEXTURE_2D);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, GL_MODULATE );

  glBindTexture(GL_TEXTURE_2D, font->getTexture());
  glTranslatef(getAbsCoor2D().x, getAbsCoor2D().y, 0);
  glRotatef(this->getAbsDir2D(), 0, 0, 1);

  Glyph* tmpGlyph;
  float posX = 0.0f;
  glBegin(GL_QUADS);
  for (unsigned int i = 0; i < this->text.size(); i++)
  {
    if(likely((tmpGlyph = this->getFont()->getGlyphArray()[this->text[i]]) != NULL))
    {
      glTexCoord2f(tmpGlyph->texCoord[1], tmpGlyph->texCoord[2]);
      glVertex2d(posX+tmpGlyph->maxX*this->getSize(), 0);

      glTexCoord2f(tmpGlyph->texCoord[1], tmpGlyph->texCoord[3]);
      glVertex2d(posX+tmpGlyph->maxX*this->getSize(), this->getSize());

      glTexCoord2f(tmpGlyph->texCoord[0], tmpGlyph->texCoord[3]);
      glVertex2d(posX+tmpGlyph->minX*this->getSize(), this->getSize());

      glTexCoord2f(tmpGlyph->texCoord[0], tmpGlyph->texCoord[2]);
      glVertex2d(posX+tmpGlyph->minX*this->getSize(), 0);

      posX += tmpGlyph->advance * this->getSize();
    }
  }
  glEnd();
  glPopMatrix();
}


/**
 * @brief setting up the Text-Width.
 */
void Text::setupTextWidth()
{
  float width = 0;
  for (unsigned int i = 0; i < this->text.size(); i++)
    if(this->font->getGlyphArray()[this->text[i]] != NULL)
      width += this->font->getGlyphArray()[this->text[i]]->advance;
  this->setSizeX2D(width *this->getSize());
}


/**
 * @brief prints out some nice debug information about this text
 */
void Text::debug() const
{
  PRINT(0)("=== TEXT: %s (with Font:'%s')  displaying %s ===\n", this->getName(), this->font->getName(), this->text.c_str());
  PRINT(0)("Color: %0.2f %0.2f %0.2f\n", this->color.x, this->color.y, this->color.z);
}

