/*!
 * @file font.h
 * brief Definition of the FONT-loading class
 *
 * !! IMPORTANT !! When using ttf fonts clear the license issues prior to
 * adding them to orxonox. This is really important, because we do not want
 * to offend anyone.
 */

#ifndef _FONT_H
#define _FONT_H

#include "texture.h"

#include "glincl.h"


#ifdef HAVE_SDL_TTF_H
#include <SDL_ttf.h>
#else
#include <SDL/SDL_ttf.h>
#endif

/* some default values */
#define FONT_NUM_COLORS              256           //!< number of colors.

#define FONT_HIGHEST_KNOWN_CHAR      128           //!< The highest character known to the textEngine.
#define FONT_DEFAULT_RENDER_SIZE     50            //!< At what Resolution to render fonts.
// FORWARD DECLARATION

//! A struct for handling glyphs
/**
 * a Glyph is one letter of a certain font
 */
struct Glyph
{
  // Glyph-specific (size and so on)
  Uint16   character;         //!< The character
  float    minX;              //!< The minimum distance from the origin in X
  float    maxX;              //!< The maximum distance from the origin in X
  float    minY;              //!< The minimum distance from the origin in Y
  float    maxY;              //!< The maximum distance from the origin in Y
  float    width;             //!< The width of the Glyph
  float    height;            //!< The height of the Glyph
  float    bearingX;          //!< How much is right of the Origin
  float    bearingY;          //!< How much is above the Origin
  float    advance;           //!< How big a Glyph would be in monospace-mode

  GLfloat texCoord[4];        //!< Texture coordinates: 0:left, 1:right, 2: top, 3: bottom.
};


//! A class to handle a Font of a certain ttf-File/image-file, Size.
class Font : public Texture
{
  public:
    Font(const std::string& fontFile,
         unsigned int renderSize);
    Font(const std::string& imageFile);
    Font(char** xpmArray);
    virtual ~Font();

    void init();

  //  font
    bool loadFontFromTTF(const std::string& fontFile);
    bool loadFontFromSDL_Surface(SDL_Surface* surface);

    void setStyle(const std::string& renderStyle);

    /** @returns a Pointer to the Array of Glyphs */
    inline Glyph** getGlyphArray() const { return this->glyphArray; };
    /** @returns the a pointer to the TTF */
    inline TTF_Font* getTTF() const { return this->fontTTF; };

    /** @returns the default Font */
    inline static Font* getDefaultFont() { if (Font::defaultFont == NULL) initDefaultFont(); return Font::defaultFont; };

    void createAsciiImage(const std::string& fileName, unsigned int size) const;
    static void initDefaultFont();
    static void removeDefaultFont();

  private:
    int getMaxHeight() const;
    int getMaxAscent() const;
    int getMaxDescent() const;
    bool getGlyphMetrics(Glyph* glyph, Uint16 character);

    bool createFastTexture();

    void initGlyphs(Uint16 from, Uint16 count);
    int findOptimalFastTextureSize();

    void debug();

  private:
    static Font*  defaultFont;         //!< a default font, that is used, if other fonts were unable to be loaded.
    // information about the Font
    TTF_Font*     fontTTF;             //!< The font we use for this.
    int           renderStyle;         //!< The Renderstyle
    unsigned int  renderSize;          //!< How big the Font should be rendered.

    Glyph**       glyphArray;          //!< An Array of all the Glyphs stored in the Array of Glyphs.
};

#endif /* _FONT_H */
