/*!
 * @file shell_completion_plugin.h
 * @brief The Shell Completion Plugin
 */

#ifndef _SHELL_COMPLETION_PLUGIN_H
#define _SHELL_COMPLETION_PLUGIN_H

#include <list>
#include <vector>
#include <string>
#include "multi_type.h"

namespace OrxShell
{
  //! The Base of All Completors
  class CompletorPlugin
  {
  public:
    virtual void addToCompleteList(std::vector<std::string>& completionList, const std::string& completionBegin) const = 0;
    virtual ~CompletorPlugin() { };

    virtual CompletorPlugin* clone() const = 0;
  protected:
    CompletorPlugin() {};
  };

  class CompletorDefault : public CompletorPlugin
  {
  public:
    CompletorDefault(const MultiType* value);
    virtual void addToCompleteList(std::vector<std::string>& completionList, const std::string& completionBegin) const;

    virtual CompletorPlugin* clone() const;
  private:
    const MultiType*    _value;
  };


  //! Completor that completes static Arrays of Strings.
  class CompletorStringArray : public CompletorPlugin
  {
  public:
    CompletorStringArray(const std::string* stringArray, unsigned int size)
        : _stringArray(stringArray), _size(size) {};
    virtual void addToCompleteList(std::vector<std::string>& completionList, const std::string& completionBegin) const;

    virtual CompletorPlugin* clone() const;
  private:
    const std::string*   _stringArray;
    unsigned int         _size;
  };


  class CompletorList : public CompletorPlugin
  {
  public:
    CompletorList(const std::list<std::string>* list);
    virtual void addToCompleteList(std::vector<std::string>& completionList, const std::string& completionBegin) const;
    virtual CompletorPlugin* clone() const;

  private:
    const std::list<std::string>* _list;
  };



  //! Completor that completes FileSystem Entries.
  class CompletorFileSystem : public CompletorPlugin
  {
  public:
    // Where to search if the completionString is empty.
    typedef enum
    {
      StartAtRoot,
      StartAtHome,
      StartAtDataDir,
    } StartDirectory;

    CompletorFileSystem(const std::string& fileExtension = "",
                        const std::string& subDir = "",
                        StartDirectory startDir = StartAtDataDir);
    virtual void addToCompleteList(std::vector<std::string>& completionList, const std::string& completionBegin) const;
    virtual CompletorPlugin* clone() const;

  private:
    std::string             _fileExtension;
    std::string             _subDir;
    StartDirectory          _startDir;
  };




  //! A Templated Completor
  template<typename CLASS> class CompletorTList : public CompletorPlugin
  {
  public:
    CompletorTList(const std::list<CLASS*>& completionList);
    virtual void addToCompleteList(std::vector<std::string>& completionList, const std::string& completionBegin)
    {};
    virtual CompletorPlugin* clone() const;
  };

}
#endif /* _SHELL_COMPLETION_PLUGIN_H */
