/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_SHELL

#include "shell_completion.h"
#include "shell_command_class.h"
#include "shell_completion_plugin.h"

#include "shell_command.h"

#include "substring.h"
#include "class_list.h"
#include "debug.h"

namespace OrxShell
{

  /**
   * @brief standard constructor
   */
  ShellCompletion::ShellCompletion()
  { }


  /**
   * @brief standard deconstructor
   */
  ShellCompletion::~ShellCompletion ()
  { }



  /**
   * @brief autocompletes the Shell's inputLine
   * @param input the input to complete, will most possibly be changed.
   * @returns true, if a result was found, false otherwise
   */
  bool ShellCompletion::autoComplete(std::string& input)
  {
    long classID = CL_NULL;                          //< the classID retrieved from the Class.
    const std::list<BaseObject*>* objectList = NULL; //< the list of Objects stored in classID's ClassList
    bool emptyComplete = false;                      //< if the completion input is empty string. e.g ""
    long completeType = NullCompletion;              //< the Type we'd like to complete.
    std::string completeString = "";                 //< the string to complete.
    unsigned int completeParam = 0;                  //< The Parameter to finish.
    const ShellCommand* command = NULL;              //< The Command.


    PRINTF(5)("AutoComplete on input\n");
    this->clearCompletionList();

    // Check if we are in a input. eg. the supplied string "class " and now we complete either function or object
    if (input.empty() || input[input.size()-1] == ' ')
      emptyComplete = true;

    // CREATE INPUTS
    SubString inputSplits(input, SubString::WhiteSpacesWithComma);

    // What String will be completed
    if (!emptyComplete && inputSplits.size() >= 1)
      completeString = inputSplits.getString(inputSplits.size()-1);

    // CLASS/ALIAS COMPLETION (on first argument)
    if (inputSplits.size() == 0 || (!emptyComplete && inputSplits.size() == 1))
    {
      completeType |= ClassCompletion;
      completeType |= AliasCompletion;
    }

    // OBJECT/FUNCTION COMPLETIONS
    else if ((emptyComplete && inputSplits.size() == 1) ||
             (!emptyComplete && inputSplits.size() == 2))
    {
      classID = ClassList::StringToID(inputSplits.getString(0));
      objectList = ClassList::getList((ClassID)classID);
      if (classID != CL_NULL)
        completeType |= ObjectCompletion;
      completeType |= FunctionCompletion;
    }
    // Complete the last Function
    else if ((emptyComplete && inputSplits.size() == 2 ) ||
             (!emptyComplete && inputSplits.size() == 3))
    {
      if (ClassList::exists(inputSplits[0], inputSplits[1]))
        completeType |= FunctionCompletion;
    }

    // Looking for ParameterCompletions.
    {
      unsigned int parameterBegin;
      if ((command = ShellCommand::getCommandFromInput(inputSplits, parameterBegin)) != NULL)
      {
        completeType |= ParamCompletion;
        if (emptyComplete)
          completeParam = inputSplits.size() - parameterBegin;
        else
          completeParam = inputSplits.size() - parameterBegin - 1;
      }
    }

    if (completeType & ClassCompletion)
      this->objectComplete(completeString, CL_SHELL_COMMAND_CLASS);
    if (completeType & ObjectCompletion)
      this->objectComplete(completeString, classID);
    if (completeType & FunctionCompletion)
      this->commandComplete(completeString, inputSplits[0]);
    if (completeType & AliasCompletion)
      this->aliasComplete(completeString);
    if (completeType & ParamCompletion)
      this->paramComplete(completeString, command, completeParam);

    this->generalComplete(input, completeString);
    return true;
  }

  /**
   * @brief autocompletes an ObjectName
   * @param objectBegin the beginning string of a Object
   * @param classID the ID of the Class to search for.
   * @return true on success, false otherwise
   */
  bool ShellCompletion::objectComplete(const std::string& objectBegin, long classID)
  {
    const std::list<BaseObject*>* boList = ClassList::getList((ClassID)classID);
    if (boList != NULL)
    {
      CompletionType type = ObjectCompletion;
      if (classID == CL_SHELL_COMMAND_CLASS)
        type = ClassCompletion;
      if (!this->addToCompleteList(*boList, objectBegin, type))
        return false;
    }
    else
      return false;
    return true;
  }

  /**
   * @brief completes a Command
   * @param commandBegin the beginning of the function String
   * @param classID the class' ID to complete the function of
   */
  bool ShellCompletion::commandComplete(const std::string& commandBegin, const std::string& className)
  {
    std::list<std::string> fktList;
    ShellCommandClass::getCommandListOfClass(className, fktList);
    if (!this->addToCompleteList(fktList, commandBegin, FunctionCompletion))
      return false;
    return true;
  }

  /**
   * @brief completes an Alias
   * @param aliasBegin the beginning of the Alias-String to complete
   * @returns true on succes, false if something went wrong
   */
  bool ShellCompletion::aliasComplete(const std::string& aliasBegin)
  {
    std::list<std::string> aliasList;
    ShellCommandAlias::getCommandListOfAlias(aliasList);
    if (!this->addToCompleteList(aliasList, aliasBegin, AliasCompletion))
      return false;
    return true;
  }

  /**
   * @brief completes Parameters.
   * @param paramBegin: Begin of the Parameters.
   * @returns true on succes, false if something went wrong
   */
  bool ShellCompletion::paramComplete(const std::string& paramBegin, const ShellCommand* command, unsigned int paramNumber)
  {
    if (paramNumber >= command->getParamCount())
    {
      PRINT(0)("Last Parameter reached\n");
      return false;
    }
    std::vector<std::string> completed;
    command->getCompletorPlugin(paramNumber)->addToCompleteList(completed, paramBegin);
    for (unsigned int i = 0; i < completed.size(); i++)
      this->completionList.push_back(CompletionElement(completed[i], ParamCompletion));
    return true;
  }


  /**
   * @brief completes the inputline on grounds of an inputList
   * @param input the Input to complete.
   * @param begin the String to search in the inputList, and to extend with it.
   * @param displayAs how to display the found value to the user, printf-style, !!with only one %s!! ex.: "::%s::"
   * @param addBack what should be added at the end of the completion
   * @param addFront what should be added to the front of one finished completion
   * @return true if ok, false otherwise
   */
  bool ShellCompletion::generalComplete(std::string& input,
                                        const std::string& begin, const std::string& displayAs,
                                        const std::string& addBack, const std::string& addFront)
  {
    if (completionList.size() == 0)
      return false;

    CompletionElement addElem = completionList.front();
    const std::string& addString = addElem.name;
    unsigned int addLength = addString.size();
    unsigned int inputLenght = begin.size();

    // Determin the longest Match (starting with the first candidate in full length).
    CompletionType changeType = NullCompletion;
    std::vector<CompletionElement>::iterator charIT;
    for (charIT = completionList.begin(); charIT != completionList.end(); charIT++)
    {
      printf("== %s\n", (*charIT).name.c_str());
      if ((*charIT).type != changeType)
      {
        if (changeType != NullCompletion)
          PRINT(0)("\n");
        PRINT(0)("%s: ", ShellCompletion::typeToString((*charIT).type).c_str());
        changeType = (*charIT).type;
      }
      PRINTF(0)("%s ", (*charIT).name.c_str());
      for (unsigned int i = inputLenght; i < addLength; i++)
        if (addString[i] != (*charIT).name[i])
          addLength = i;
    }
    PRINT(0)("\n");

    if (addLength >= inputLenght)
    {
      std::string adder = addString;
      adder.resize(addLength);

      input.resize(input.size()-inputLenght);
      input += adder;

      if (completionList.size() == 1)
      {
        if ( addBack != "")
          input += addBack;
        input += ' ';
      }
    }
    return true;
  }

  /**
   * @brief searches for classes, which beginn with completionBegin
   * @param inputList the List to parse through
   * @param completionBegin the beginning string
   * !! The strings MUST NOT be deleted !!
   */
  bool ShellCompletion::addToCompleteList(const std::list<std::string>& inputList, const std::string& completionBegin, CompletionType type)
  {
    unsigned int searchLength = completionBegin.size();

    std::list<std::string>::const_iterator string;
    for (string = inputList.begin(); string != inputList.end(); string++)
    {
      if ((*string).size() >= searchLength &&
          !nocaseCmp(*string, completionBegin, searchLength))
      {
        this->completionList.push_back(CompletionElement (*string, type));
      }
    }
    return true;
  }

  /**
   * @brief searches for classes, which beginn with completionBegin
   * @param inputList the List to parse through
   * @param completionBegin the beginning string
   * !! The strings MUST NOT be deleted !!
   */
  bool ShellCompletion::addToCompleteList(const std::list<BaseObject*>& inputList, const std::string& completionBegin, CompletionType type)
  {
    unsigned int searchLength = completionBegin.size();

    std::list<BaseObject*>::const_iterator bo;
    for(bo = inputList.begin(); bo != inputList.end(); bo++)
    {
      if ((*bo)->getName() != NULL &&
          strlen((*bo)->getName()) >= searchLength &&
          !nocaseCmp((*bo)->getName(), completionBegin, searchLength))
      {
        this->completionList.push_back(CompletionElement((*bo)->getName(), type));
      }
    }

    return true;
  }

  /**
   * @brief deletes the Completion List.
   *
   * This is done at the beginning of each completion-run
   */
  void ShellCompletion::clearCompletionList()
  {
    this->completionList.clear();
  }

  const std::string& ShellCompletion::typeToString(CompletionType type)
  {
    switch (type)
    {
        default:// SHELLC_NONE
        return typeNames[0];
        case  ClassCompletion:
        return typeNames[1];
        case ObjectCompletion:
        return typeNames[2];
        case FunctionCompletion:
        return typeNames[3];
        case AliasCompletion:
        return typeNames[4];
        case ParamCompletion:
        return typeNames[5];
    }
  }


  const std::string ShellCompletion::typeNames[] =
    {
      "error",
      "class",
      "object",
      "function",
      "alias",
      "parameter",
    };

}
