/*!
 * @file shell_input.h
 * @brief Shell Input is an InputLine for the Shell.
 *
 * @todo move around in the InputText with the cursor (SDLK_LEFT/SDLK_RIGHT)
 * @todo blinking curson (can blink with the speed of the RepeatDelay)
*/

#ifndef _SHELL_INPUT_H
#define _SHELL_INPUT_H

#include "text.h"
#include "event_listener.h"
#include "shell_completion.h"

#include <list>

namespace OrxShell
{
  // FORWARD DECLARATION
  class ShellCompletion;

  //! An InputLine for the Shell
  /**
   * The ShellInput has the ability to catch and display user input.
   * The ShellInput is auto-completed after the user presses [TAB]
   * The ShellInput is executed (and sent back to the Application) on Pressing [ENTER]
   * [UP] and [DOWN] move through the history of allready given commands.
   */
  class ShellInput : public Text,  public EventListener
  {

  public:
    ShellInput();
    virtual ~ShellInput();

    /** @returns the inputLine */
    const std::string& getInput() const { return this->inputLine; };

    // InputLine
    void setInputText(const std::string& text);
    void addCharacter(char character);
    void addCharacters(const std::string& characters);
    void removeCharacters(unsigned int characterCount = 1);
    void flush();

    void setRepeatDelay(float repeatDelay, float repeatRate);

    bool executeCommand();

    /** sets the count of the History's entries */
    void setHistoryLength(unsigned int historyLength) { this->historyLength = historyLength; };
    void historyMoveUp();
    void historyMoveDown();

    void help(const std::string& className = "", const std::string& function = "");

    virtual void tick(float dt);
    virtual void process(const Event &event);

  private:
    // HANDLING TEXT INPUT
    ShellCompletion                   completion;       //!< The Completion Interface.

    std::string                       inputLine;        //!< the Char-Array of the Buffer
    float                             repeatRate;       //!< The Repeat-Delay.
    float                             repeatDelay;      //!< The delay of the first Character of a given Character.
    float                             delayed;          //!< how much of the delay is remaining.
    Uint16                            pressedKey;       //!< the pressed key that will be repeated.

    std::list<std::string>            history;          //!< The history of given commands.
    std::list<std::string>::iterator  historyIT;        //!< The locator that tells us, where we are in the history.
    unsigned int                      historyLength;    //!< The maximum length of the InputHistory.
    bool                              historyScrolling; //!< true if we are scrolling through the history.
  };

}

#endif /* _SHELL_INPUT_H */
