/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_SHELL

#include "shell_command_class.h"

#include "shell_command.h"

#include "debug.h"
#include "class_list.h"
#include "compiler.h"

namespace OrxShell
{
  std::vector<ShellCommandClass*> ShellCommandClass::commandClassList;

  /**
   * creates a new ShellCommandClass
   * @param className the Name of the command-class to create
   */
  ShellCommandClass::ShellCommandClass(const std::string& className)
      : className(className)
  {
    this->setClassID(CL_SHELL_COMMAND_CLASS, "ShellCommandClass");
    this->setName(className);

    this->classID = CL_NULL;

    ShellCommandClass::commandClassList.push_back(this);
  }

  /**
   * destructs the shellCommandClass again
   */
  ShellCommandClass::~ShellCommandClass()
  {
    while(!this->commandList.empty())
      delete this->commandList.back();
  }

  /**
   * @param command the Command to register.
   */
  void ShellCommandClass::registerCommand(ShellCommand* command)
  {
    this->commandList.push_back(command);
  }

  /**
   * @brief unregister command.
   * @param command the Command to unregister.
   */
  void ShellCommandClass::unregisterCommand(ShellCommand* command)
  {
    std::vector<ShellCommand*>::iterator delC = std::find(this->commandList.begin(), this->commandList.end(), command);
    if (delC != this->commandList.end())
      this->commandList.erase(delC);
  }

  /**
   * @brief unregisters all Commands that exist
   */
  void ShellCommandClass::unregisterAllCommands()
  {
    // unregister all commands and Classes
    std::vector<ShellCommandClass*>::iterator classIT;
    for (classIT = ShellCommandClass::commandClassList.begin(); classIT != ShellCommandClass::commandClassList.end(); classIT++)
      delete (*classIT);
  }


  /**
   * @brief collects the Commands registered to some class.
   * @param className the name of the Class to collect the Commands from.
   * @param stringList a List to paste the Commands into.
   * @returns true on success, false otherwise
   */
  bool ShellCommandClass::getCommandListOfClass(const std::string& className, std::list<std::string>& stringList)
  {
    std::vector<ShellCommandClass*>::iterator elem;
    for(elem = ShellCommandClass::commandClassList.begin(); elem != ShellCommandClass::commandClassList.end(); elem++)
    {
      if (className == (*elem)->getName())
      {
        std::vector<ShellCommand*>::iterator command;
        for(command = (*elem)->commandList.begin(); command != (*elem)->commandList.end(); command++)
          stringList.push_back((*command)->getName());
      }
    }
    return true;
  }


  /**
   * @brief checks if a Class is already registered to the Commands' class-stack
   * @param className the Name of the Class to check for
   * @returns the CommandClass if found, NULL otherwise
   */
  const ShellCommandClass* ShellCommandClass::exists(const std::string& className)
  {
    std::vector<ShellCommandClass*>::const_iterator classIT;
    for (classIT = ShellCommandClass::commandClassList.begin(); classIT != ShellCommandClass::commandClassList.end(); classIT++)
    {
      if (className == (*classIT)->className)
      {
        if ((*classIT)->classID == CL_NULL)
          (*classIT)->classID = ClassList::StringToID(className);

        return (*classIT);
      }
    }
    return NULL;
  }

  /**
   * @brief searches for a CommandClass
   * @param className the name of the CommandClass
   * @returns the CommandClass if found, or a new CommandClass if not
   */
  ShellCommandClass* ShellCommandClass::getCommandClass(const std::string& className)
  {
    std::vector<ShellCommandClass*>::iterator classIT;
    for (classIT = ShellCommandClass::commandClassList.begin(); classIT != ShellCommandClass::commandClassList.end(); classIT++)
      if (className == (*classIT)->className)
        return (*classIT);
    return new ShellCommandClass(className);
  }

  /**
   * @brief displays help about ShellCommandClass
   * @param className: the Class of Commands to show help about
   */
  void ShellCommandClass::help(const std::string& className)
  {
    std::vector<ShellCommandClass*>::iterator classIT;
    for (classIT = ShellCommandClass::commandClassList.begin(); classIT != ShellCommandClass::commandClassList.end(); classIT++)
    {
      if (className == (*classIT)->className)
      {
        PRINT(0)("Class:'%s' registered %d commands: \n", (*classIT)->className.c_str(), (*classIT)->commandList.size());
        std::vector<ShellCommand*>::const_iterator cmdIT;
        for (cmdIT = (*classIT)->commandList.begin(); cmdIT != (*classIT)->commandList.end(); cmdIT++)
        {
          PRINT(0)("  command:'%s' : params:%d: ", (*cmdIT)->getName(), (*cmdIT)->executor->getParamCount());
          /// FIXME
          /*          for (unsigned int i = 0; i< elem->paramCount; i++)
            PRINT(0)("%s ", ShellCommand::paramToString(elem->parameters[i]));*/
          if (!(*cmdIT)->description.empty())
            PRINT(0)("- %s", (*cmdIT)->description.c_str());
          PRINT(0)("\n");
        }
        return;
      }
    }
    PRINTF(3)("Class %s not found in Command's classes\n", className.c_str());
  }

}




