/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_SHELL

#include "shell_command_class.h"

#include "shell_command.h"

#include "debug.h"
#include "class_list.h"
#include "compiler.h"

#include <stdio.h>
#include <string.h>

namespace OrxShell
{

  std::list<ShellCommandClass*>* ShellCommandClass::commandClassList = NULL;
  std::list<ShellCommandAlias*>* ShellCommandClass::aliasList = NULL;

  /**
   * creates a new ShellCommandClass
   * @param className the Name of the command-class to create
   */
  ShellCommandClass::ShellCommandClass(const std::string& className)
      : className(className)
  {
    this->setClassID(CL_SHELL_COMMAND_CLASS, "ShellCommandClass");
    this->setName(className);

    this->classID = CL_NULL;

    ShellCommandClass::commandClassList->push_back(this);
  }

  /**
   * destructs the shellCommandClass again
   */
  ShellCommandClass::~ShellCommandClass()
  {
    while(this->commandList.size() > 0)
    {
      delete this->commandList.front();
      this->commandList.pop_front();
    }
  }

  /**
   * collects the Commands registered to some class.
   * @param className the name of the Class to collect the Commands from.
   * @param stringList a List to paste the Commands into.
   * @returns true on success, false otherwise
   */
  bool ShellCommandClass::getCommandListOfClass(const std::string& className, std::list<std::string>& stringList)
  {
    std::list<ShellCommandClass*>::iterator elem;
    for(elem = ShellCommandClass::commandClassList->begin(); elem != ShellCommandClass::commandClassList->end(); elem++)
    {
      if (className == (*elem)->getName())
      {
        std::list<ShellCommand*>::iterator command;
        for(command = (*elem)->commandList.begin(); command != (*elem)->commandList.end(); command++)
          stringList.push_back((*command)->getName());
      }
    }
    return true;
  }

  /**
   * collects the Aliases registered to the ShellCommands
   * @param stringList a List to paste the Aliases into.
   * @returns true on success, false otherwise
   */
  bool ShellCommandClass::getCommandListOfAlias(std::list<std::string>& stringList)
  {
    if (ShellCommandClass::aliasList == NULL)
      return false;

    std::list<ShellCommandAlias*>::iterator alias;
    for (alias = ShellCommandClass::aliasList->begin(); alias != ShellCommandClass::aliasList->end(); alias++)
      stringList.push_back((*alias)->getName());
    return true;
  }

  /**
   * unregisters all Commands that exist
   */
  void ShellCommandClass::unregisterAllCommands()
  {
    if (ShellCommandClass::commandClassList != NULL)
    {
      // unregister all commands and Classes
      std::list<ShellCommandClass*>::iterator classIT;
      for (classIT = ShellCommandClass::commandClassList->begin(); classIT != ShellCommandClass::commandClassList->end(); classIT++)
        delete (*classIT);
      delete ShellCommandClass::commandClassList;
      ShellCommandClass::commandClassList = NULL;
    }

    // unregister all aliases (there should be nothing to do here :))
    if (ShellCommandClass::aliasList != NULL)
    {
      std::list<ShellCommandAlias*>::iterator alias;
      for (alias = ShellCommandClass::aliasList->begin(); alias != ShellCommandClass::aliasList->end(); alias++)
        delete (*alias);
      delete ShellCommandClass::aliasList;
      ShellCommandClass::aliasList = NULL;
    }
  }

  /**
   * checks if a Class is already registered to the Commands' class-stack
   * @param className the Name of the Class to check for
   * @returns the CommandClass if found, NULL otherwise
   */
  const ShellCommandClass* ShellCommandClass::isRegistered(const std::string& className)
  {
    if (ShellCommandClass::commandClassList == NULL)
      initCommandClassList();

    std::list<ShellCommandClass*>::const_iterator classIT;
    for (classIT = ShellCommandClass::commandClassList->begin(); classIT != ShellCommandClass::commandClassList->end(); classIT++)
    {
      if (className == (*classIT)->className)
      {
        if ((*classIT)->classID == CL_NULL)
          (*classIT)->classID = ClassList::StringToID(className);

        return (*classIT);
      }
    }
    return NULL;
  }

  /**
   * searches for a CommandClass
   * @param className the name of the CommandClass
   * @returns the CommandClass if found, or a new CommandClass if not
   */
  ShellCommandClass* ShellCommandClass::getCommandClass(const std::string& className)
  {
    if (ShellCommandClass::commandClassList == NULL)
      initCommandClassList();

    std::list<ShellCommandClass*>::iterator classIT;
    for (classIT = ShellCommandClass::commandClassList->begin(); classIT != ShellCommandClass::commandClassList->end(); classIT++)
    {
      if (className == (*classIT)->className)
      {
        return (*classIT);
      }
    }
    return new ShellCommandClass(className);
  }

  /**
   * @brief initializes the CommandList (if it is NULL)
   */
  void ShellCommandClass::initCommandClassList()
  {
    if (ShellCommandClass::commandClassList == NULL)
    {
      ShellCommandClass::commandClassList = new std::list<ShellCommandClass*>;
      ShellCommand::registerCommand("debug", "ShellCommand", ExecutorStatic<ShellCommand>(ShellCommand::debug));
    }
  }

  /**
   * @brief displays help about ShellCommandClass
   * @param className: the Class of Commands to show help about
   */
  void ShellCommandClass::help(const std::string& className)
  {
    if (likely(ShellCommandClass::commandClassList != NULL))
    {
      std::list<ShellCommandClass*>::iterator classIT;
      for (classIT = ShellCommandClass::commandClassList->begin(); classIT != ShellCommandClass::commandClassList->end(); classIT++)
      {
        if (className == (*classIT)->className)
        {
          PRINT(0)("Class:'%s' registered %d commands: \n", (*classIT)->className.c_str(), (*classIT)->commandList.size());
          std::list<ShellCommand*>::const_iterator cmdIT;
          for (cmdIT = (*classIT)->commandList.begin(); cmdIT != (*classIT)->commandList.end(); cmdIT++)
          {
            PRINT(0)("  command:'%s' : params:%d: ", (*cmdIT)->getName(), (*cmdIT)->executor->getParamCount());
            /// FIXME
            /*          for (unsigned int i = 0; i< elem->paramCount; i++)
              PRINT(0)("%s ", ShellCommand::paramToString(elem->parameters[i]));*/
            if (!(*cmdIT)->description.empty())
              PRINT(0)("- %s", (*cmdIT)->description.c_str());
            PRINT(0)("\n");
          }
          return;
        }
      }
      PRINTF(3)("Class %s not found in Command's classes\n", className.c_str());
    }
    else
    {
      PRINTF(1)("List of commandClasses does not exist");
    }
  }

}
