/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_SHELL

#include "shell_command.h"
#include "shell_command_class.h"

#include "compiler.h"
#include "debug.h"
#include "class_list.h"

#include "key_names.h"
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

namespace OrxShell
{

  /**
   * constructs and registers a new Command
   * @param commandName the name of the Command
   * @param className the name of the class to apply this command to
   * @param paramCount the count of parameters this command takes
   */
  ShellCommand::ShellCommand(const std::string& commandName, const std::string& className, const Executor& executor)
  {
    this->setClassID(CL_SHELL_COMMAND, "ShellCommand");
    PRINTF(5)("create shellcommand %s %s\n", commandName, className);
    this->setName(commandName);
    this->executor = executor.clone();
    this->executor->setName(commandName);

    //  this->classID = classID;
    this->shellClass = ShellCommandClass::getCommandClass(className); //ClassList::IDToString(classID);
    if (this->shellClass != NULL)
      this->shellClass->commandList.push_back(this);
  }

  /**
   * deconstructs a ShellCommand
   */
  ShellCommand::~ShellCommand()
  {
    if (this->alias != NULL && ShellCommandClass::aliasList != NULL)
    {
      ShellCommandClass::aliasList->remove(this->alias);
      delete this->alias;
    }
    delete this->executor;
  }

  /**
   * registers a new ShellCommand
   */
  ShellCommand* ShellCommand::registerCommand(const std::string& commandName, const std::string& className, const Executor& executor)
  {
    if (ShellCommand::isRegistered(commandName, className))
      return NULL;
    else
      return new ShellCommand(commandName, className, executor);

  }

  /**
   * unregister an existing commandName
   * @param className the name of the Class the command belongs to.
   * @param commandName the name of the command itself
   */
  void ShellCommand::unregisterCommand(const std::string& commandName, const std::string& className)
  {
    /// FIXME
    /*  if (ShellCommandClass::commandClassList == NULL)
        ShellCommandClass::initCommandClassList();

     const ShellCommandClass* checkClass = ShellCommandClass::isRegistered(className);

     if (checkClass != NULL)
      {
        std::list<ShellCommand*>::iterator elem;
        for (elem = checkClass->commandList.begin(); elem != checkClass->commandList.end(); elem++)
        {
          if (!strcmp(commandName, (*elem)->getName()))
          {
            delete (*elem);
            checkClass->commandList.remove(*elem);
            break;
          }
        }

        if (checkClass->commandList->size() == 0)
        {
          ShellCommandClass::commandClassList->remove(checkClass);
          delete checkClass;
        }
      }*/
  }

  /**
   * checks if a command has already been registered.
   * @param commandName the name of the Command
   * @param className the name of the Class the command should apply to.
   * @returns true, if the command is registered/false otherwise
   *
   * This is used internally, to see, if we have multiple command subscriptions.
   * This is checked in the registerCommand-function.
   */
  bool ShellCommand::isRegistered(const std::string& commandName, const std::string& className)
  {
    if (ShellCommandClass::commandClassList == NULL)
    {
      ShellCommandClass::initCommandClassList();
      return false;
    }

    const ShellCommandClass* checkClass = ShellCommandClass::isRegistered(className);
    if (checkClass != NULL)
    {
      std::list<ShellCommand*>::const_iterator elem;
      for (elem = checkClass->commandList.begin(); elem != checkClass->commandList.end(); elem++)
      {
        if (commandName == (*elem)->getName())
        {
          PRINTF(2)("Command '%s::%s' already registered\n", className.c_str(), commandName.c_str());
          return true;
        }
      }
      return false;
    }
    else
      return false;
  }


  /**
   * executes commands
   * @param executionString the string containing the following input
   * ClassName [ObjectName] functionName [parameter1[,parameter2[,...]]]
   * @return true on success, false otherwise.
   */
  bool ShellCommand::execute(const std::string& executionString)
  {
    if (ShellCommandClass::commandClassList == NULL)
      return false;

    long classID = CL_NULL;                      //< the classID retrieved from the Class.
    ShellCommandClass* commandClass = NULL;      //< the command class this command applies to.
    const std::list<BaseObject*>* objectList = NULL;   //< the list of Objects stored in classID
    BaseObject* objectPointer = NULL;            //< a pointer to th Object to Execute the command on
    bool emptyComplete = false;                  //< if the completion input is empty string. e.g ""
    //  long completeType = SHELLC_NONE;           //< the Type we'd like to complete.
    SubString inputSplits(executionString, SubString::WhiteSpacesWithComma);


    // if we do not have any input return
    if (inputSplits.empty())
      return false;

    // if we only have one input (!MUST BE AN ALIAS)
    if (inputSplits.size() >= 1)
    {
      // CHECK FOR ALIAS
      if (ShellCommandClass::aliasList != NULL)
      {
        std::list<ShellCommandAlias*>::iterator alias;
        for (alias = ShellCommandClass::aliasList->begin(); alias != ShellCommandClass::aliasList->end(); alias++ )
        {
          if (inputSplits.getString(0) == (*alias)->getName() && (*alias)->getCommand() != NULL &&
              (*alias)->getCommand()->shellClass != NULL )
          {
            objectList = ClassList::getList((*alias)->getCommand()->shellClass->getName());
            if (objectList != NULL)
            {
              (*(*alias)->getCommand()->executor)(objectList->front(), inputSplits.getSubSet(1).join()); /// TODO CHECK IF OK
              return true;
            }
            /// TODO CHECK FOR STATIC functions.
          }
        }
      }

      // looking for a Matching Class
      if (likely(ShellCommandClass::commandClassList != NULL))
      {
        std::list<ShellCommandClass*>::iterator commandClassIT;
        for (commandClassIT = ShellCommandClass::commandClassList->begin(); commandClassIT != ShellCommandClass::commandClassList->end(); commandClassIT++)
        {
          if ((*commandClassIT)->getName() && inputSplits[0] == (*commandClassIT)->getName())
          {
            //elemCL->getName();
            classID = ClassList::StringToID((*commandClassIT)->getName());
            commandClass = (*commandClassIT);
            objectList = ClassList::getList((ClassID)classID);
            break;
          }
        }
      }

      // Second Agument. (either Object, or Function)
      if (commandClass != NULL && inputSplits.size() >= 2)
      {
        int fktPos = 1; // The position of the Function (either at pos 1, or 2)
        // If we have an ObjectList.
        if (objectList != NULL)
        {
          // Checking for a Match in the Objects of classID (else take the first)
          std::list<BaseObject*>::const_iterator object;
          for (object = objectList->begin(); object != objectList->end(); object++)
          {
            if ((*object)->getName() != NULL && inputSplits[1] == (*object)->getName())
            {
              objectPointer = (*object);
              fktPos = 2;
              break;
            }
          }

          // if we did not find an Object with matching name, take the first.
          if (objectPointer == NULL)
            objectPointer = objectList->front();
        }

        // match a function.
        if (commandClass != NULL && (fktPos == 1 || (fktPos == 2 && inputSplits.size() >= 3)))
        {
          std::list<ShellCommand*>::iterator cmdIT;
          for (cmdIT = commandClass->commandList.begin(); cmdIT != commandClass->commandList.end(); cmdIT++)
          {
            if (inputSplits[fktPos] == (*cmdIT)->getName())
            {
              if (objectPointer == NULL && (*cmdIT)->executor->getType() & Executor_Objective)
                return false;
              else
              {
                (*(*cmdIT)->executor)(objectPointer, inputSplits.getSubSet(fktPos+1).join()); /// TODO CHECK IF OK
                return true;
              }
            }
          }
        }
      }
    }
    return false;
  }

  /**
   * lets a command be described
   * @param description the description of the Given command
   */
  ShellCommand* ShellCommand::describe(const std::string& description)
  {
    if (this == NULL)
      return NULL;
    else
    {
      this->description = description;
      return this;
    }
  }

  /**
   * adds an Alias to this Command
   * @param alias the name of the Alias to set
   * @returns itself
   */
  ShellCommand* ShellCommand::setAlias(const std::string& alias)
  {
    if (this == NULL)
      return NULL;

    if (this->alias != NULL)
    {
      PRINTF(2)("not more than one Alias allowed for functions (%s::%s)\n", this->getName(), this->shellClass->getName());
    }
    else
    {
      if (ShellCommandClass::aliasList == NULL)
        ShellCommandClass::aliasList = new std::list<ShellCommandAlias*>;

      ShellCommandAlias* aliasCMD = new ShellCommandAlias(alias, this);
      ShellCommandClass::aliasList->push_back(aliasCMD);
      this->alias = aliasCMD;
    }
    return this;
  }

  /**
   * @brief set the default values of the executor
   * @param value0 the first default value
   * @param value1 the second default value
   * @param value2 the third default value
   * @param value3 the fourth default value
   * @param value4 the fifth default value
   */
  ShellCommand* ShellCommand::defaultValues(const MultiType& value0, const MultiType& value1,
      const MultiType& value2, const MultiType& value3,
      const MultiType& value4)
  {
    if (this == NULL || this->executor == NULL)
      return NULL;

    this->executor->defaultValues(value0, value1, value2, value3, value4);

    return this;
  }

  /**
   * prints out nice information about the Shells Commands
   */
  void ShellCommand::debug()
  {
    if (ShellCommandClass::commandClassList == NULL)
    {
      PRINT(0)("No Command registered.\n");
      return;
    }

    std::list<ShellCommandClass*>::iterator classIT;
    for (classIT = ShellCommandClass::commandClassList->begin(); classIT != ShellCommandClass::commandClassList->end(); classIT++)
    {
      PRINT(0)("Class:'%s' registered %d commands: \n", (*classIT)->className.c_str(), (*classIT)->commandList.size());

      std::list<ShellCommand*>::iterator cmdIT;
      for (cmdIT = (*classIT)->commandList.begin(); cmdIT != (*classIT)->commandList.end(); cmdIT++)
      {
        PRINT(0)("  command:'%s' : params:%d: ", (*cmdIT)->getName(), (*cmdIT)->executor->getParamCount());
        /// FIXME
        /*      for (unsigned int i = 0; i< elem->paramCount; i++)
         printf("%s ", ShellCommand::paramToString(elem->parameters[i]));*/
        if (!(*cmdIT)->description.empty())
          printf("- %s", (*cmdIT)->description.c_str());
        printf("\n");

      }
    }
  }

  /**
   * converts a Parameter to a String
   * @param parameter the Parameter we have.
   * @returns the Name of the Parameter at Hand
   */
  const char* ShellCommand::paramToString(long parameter)
  {
    return MultiType::MultiTypeToString((MT_Type)parameter);
  }

}
