/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "shell_completion.h"
#include "shell_command_class.h"

#include "shell_input.h"
#include "shell_command.h"

#include "substring.h"
#include "base_object.h"
#include "class_list.h"
#include "debug.h"

#include "stdlibincl.h"

using namespace std;

/**
 * @brief standard constructor
 */
ShellCompletion::ShellCompletion()
{ }


/**
 * @brief standard deconstructor
 */
ShellCompletion::~ShellCompletion ()
{ }



/**
 * @brief autocompletes the Shell's inputLine
 * @returns true, if a result was found, false otherwise
 */
bool ShellCompletion::autoComplete(std::string& input)
{
  const char* completionLine;           //< the inputLine we complete.

  long classID;                         //< the classID retrieved from the Class.
  const std::list<BaseObject*>* objectList;   //< the list of Objects stored in classID
  bool emptyComplete = false;           //< if the completion input is empty string. e.g ""
  long completeType = SHELLC_NONE;      //< the Type we'd like to complete.
  const char* completeString;           //< the string to complete.


  PRINTF(5)("AutoComplete on input\n");
  this->emptyCompletionList();

  // Check if we are in a input. eg. the supplied string "class " and now we complete either function or object
  if (input[input.size()-1] == ' ')
  {
    emptyComplete = true;
  }

  // CREATE INPUTS
  SubString inputSplits(input, " \t\n,");

  // What String will be completed
  if (emptyComplete == true)
    completeString = "";
  else
    completeString = inputSplits.getString(inputSplits.size()-1).c_str();

  // CLASS COMPLETION
  if (inputSplits.size() == 0)
  {
    completeType |= SHELLC_CLASS;
    completeType |= SHELLC_ALIAS;
  }
  else if (inputSplits.size() == 1 && emptyComplete == false)
  {
    completeType |= SHELLC_CLASS;
    completeType |= SHELLC_ALIAS;
  }

  // OBJECT/FUNCTION COMPLETIONS
  else if ((inputSplits.size() == 1 && emptyComplete == true) ||
            (inputSplits.size() == 2 && emptyComplete == false))
  {
    classID = ClassList::StringToID(inputSplits.getString(0).c_str()); //FIXME
    objectList = ClassList::getList((ClassID)classID);
    if (classID != CL_NULL)
      completeType |= SHELLC_OBJECT;
    //if (objectList != NULL && objectList->getSize() == 1)
      completeType |= SHELLC_FUNCTION;
  }
  else if ((inputSplits.size() == 2 && emptyComplete == true) ||
            (inputSplits.size() == 3 && emptyComplete == false))
  {
    classID = ClassList::StringToID(inputSplits.getString(0) .c_str()); // FIXME
    if (classID == CL_NULL)
      return false;
    else
     completeType |= SHELLC_FUNCTION;
  }

  if (completeType & SHELLC_CLASS)
    this->objectComplete(completeString, CL_SHELL_COMMAND_CLASS);
  if (completeType & SHELLC_OBJECT)
    this->objectComplete(completeString, classID);
  if (completeType & SHELLC_FUNCTION)
    this->functionComplete(completeString, inputSplits.getString(0).c_str()); // FIXME
  if (completeType & SHELLC_ALIAS)
    this->aliasComplete(completeString);


  this->generalComplete(input, completeString);
  return true;
}

/**
 * @brief autocompletes a className
 * @param classBegin the Beginning of a String to autoComplete
 * @return true on success, false otherwise
 */
bool ShellCompletion::classComplete(const std::string& classBegin)
{
  const std::list<std::string>* clList = ClassList::getClassNames();
  if (clList != NULL)
  {
    if (!this->addToCompleteList(clList, classBegin, SHELLC_CLASS))
      return false;
  }
  else
    return false;
  return true;
}

/**
 * @brief autocompletes an ObjectName
 * @param objectBegin the beginning string of a Object
 * @param classID the ID of the Class to search for.
 * @return true on success, false otherwise
 */
bool ShellCompletion::objectComplete(const std::string& objectBegin, long classID)
{
  const std::list<BaseObject*>* boList = ClassList::getList((ClassID)classID);
  if (boList != NULL)
  {
    SHELLC_TYPE type = SHELLC_OBJECT;
    if (classID == CL_SHELL_COMMAND_CLASS)
      type = SHELLC_CLASS;
    if (!this->addToCompleteList(boList, objectBegin, type))
      return false;
  }
  else
    return false;
  return true;
}

/**
 * @brief completes a Function
 * @param functionBegin the beginning of the function String
 * @param classID the class' ID to complete the function of
 */
bool ShellCompletion::functionComplete(const std::string& functionBegin, const std::string& className)
{
  std::list<std::string> fktList;
  ShellCommandClass::getCommandListOfClass(className, &fktList);
  //printf("%s\n", boList->firstElement()->getName());
  if (!this->addToCompleteList(&fktList, functionBegin, SHELLC_FUNCTION))
    return false;
  return true;
}

/**
 * @brief completes an Alias
 * @param aliasBegin the beginning of the Alias-String to complete
 * @returns true on succes, false if something went wrong
 */
bool ShellCompletion::aliasComplete(const std::string& aliasBegin)
{
  std::list<std::string> aliasList;
  ShellCommandClass::getCommandListOfAlias(&aliasList);
  //printf("%s\n", boList->firstElement()->getName());
  if (!this->addToCompleteList(&aliasList, aliasBegin, SHELLC_ALIAS))
    return false;
  return true;
}


/**
 * @brief completes the inputline on grounds of an inputList
 * @param begin the String to search in the inputList, and to extend with it.
 * @param displayAs how to display the found value to the user, printf-style, !!with only one %s!! ex.: "::%s::"
 * @param addBack what should be added at the end of the completion
 * @param addFront what should be added to the front of one finished completion
 * @return true if ok, false otherwise
 */
bool ShellCompletion::generalComplete(std::string& input,
                                      const std::string& begin, const std::string& displayAs,
                                      const std::string& addBack, const std::string& addFront)
{
  if (completionList.size() == 0)
    return false;

  ShellC_Element addElem = completionList.front();
  const std::string& addString = addElem.name;
  unsigned int addLength = 0;
  unsigned int inputLenght = begin.size();

  // Determin the longest Match
  addLength = addString.size();

  SHELLC_TYPE changeType = SHELLC_NONE;
  list<ShellC_Element>::iterator charIT;
  for (charIT = completionList.begin(); charIT != completionList.end(); charIT++)
  {
    if ((*charIT).type != changeType)
    {
      if (changeType != SHELLC_NONE)
        PRINT(0)("\n");
      PRINT(0)("%s: ", ShellCompletion::typeToString((*charIT).type));
      changeType = (*charIT).type;
    }
    PRINTF(0)("%s ", (*charIT).name.c_str());
    for (unsigned int i = inputLenght; i < addLength; i++)
      if (addString[i] != (*charIT).name[i])
      {
       addLength = i;
//       break;
      }
  }
  PRINT(0)("\n");

  if (addLength >= inputLenght)
  {
    std::string adder = addString;
    adder.resize(addLength);

    input.resize(input.size()-inputLenght);
    input += adder;

    if (completionList.size() == 1)
    {
      if ( addBack != "")
       input += addBack;
      input += ' ';
    }
  }
  return true;
}

/**
 * @brief searches for classes, which beginn with completionBegin
 * @param inputList the List to parse through
 * @param completionBegin the beginning string
 * !! The strings MUST NOT be deleted !!
 */
bool ShellCompletion::addToCompleteList(const std::list<std::string>* inputList, const std::string& completionBegin, SHELLC_TYPE type)
{
  if (inputList == NULL)
    return false;
  unsigned int searchLength = completionBegin.size();

  list<std::string>::const_iterator string;
  for (string = inputList->begin(); string != inputList->end(); string++)
  {
    if ((*string).size() >= searchLength &&
          !strncasecmp((*string).c_str(), completionBegin.c_str(), searchLength))
    {
      ShellC_Element newElem;
      newElem.name = (*string).c_str();
      newElem.type = type;
      this->completionList.push_back(newElem);
    }
  }
  return true;
}

/**
 * @brief searches for classes, which beginn with completionBegin
 * @param inputList the List to parse through
 * @param completionBegin the beginning string
 * !! The strings MUST NOT be deleted !!
 */
bool ShellCompletion::addToCompleteList(const std::list<BaseObject*>* inputList, const std::string& completionBegin, SHELLC_TYPE type)
{
  if (inputList == NULL)
    return false;
  unsigned int searchLength = completionBegin.size();

  list<BaseObject*>::const_iterator bo;
  for(bo = inputList->begin(); bo != inputList->end(); bo++)
  {
    if ((*bo)->getName() != NULL &&
        strlen((*bo)->getName()) >= searchLength &&
          !strncasecmp((*bo)->getName(), completionBegin.c_str(), searchLength))
    {
      ShellC_Element newElem;
      newElem.name = (*bo)->getName();
      newElem.type = type;
      this->completionList.push_back(newElem);
    }
  }

  return true;
}

/**
 * @brief deletes the Completion List.
 *
 * This is done at the beginning of each completion-run
 */
void ShellCompletion::emptyCompletionList()
{
  this->completionList.clear();
}

const char* ShellCompletion::typeToString(SHELLC_TYPE type)
{
  switch (type)
  {
    default:// SHELLC_NONE
      return "error";
    case  SHELLC_CLASS:
      return "class";
    case SHELLC_OBJECT:
      return "object";
    case SHELLC_FUNCTION:
      return "function";
    case SHELLC_ALIAS:
      return "alias";
  }
}
