/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

//#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_

#include "shell_input.h"



#include "shell_command.h"
#include "shell_command_class.h"
#include "shell_completion.h"
#include "event_handler.h"

#include "debug.h"
#include "compiler.h"
#include "stdlibincl.h"
#include "key_names.h"


using namespace std;

SHELL_COMMAND(help, ShellInput, help)
    ->describe("retrieve some help about the input mode")
    ->setAlias("help");

/**
 * constructor
 * this also generates a ShellCompletion automatically.
*/
ShellInput::ShellInput () : Text ("")
{
  this->pressedKey = SDLK_FIRST;
  this->setClassID(CL_SHELL_INPUT, "ShellInput");

  this->inputLine = "";
  this->historyIT = this->history.begin();
  this->setHistoryLength(50);
  this->historyScrolling = false;
  this->delayed = 0;
  this->setRepeatDelay(.3, .05);

  // subscribe all keyboard commands to ES_SEHLL
  EventHandler* evh = EventHandler::getInstance();
  for (int i = 1; i < SDLK_LAST; i++)
  {
    if (!evh->isSubscribed(ES_SHELL, i))
      evh->subscribe(this, ES_SHELL, i);
  }
  // unsubscribe unused TODO improve.
  evh->unsubscribe(ES_SHELL, SDLK_BACKQUOTE);
  evh->unsubscribe(ES_SHELL, SDLK_F12);
  evh->unsubscribe(ES_SHELL, SDLK_PAGEUP);
  evh->unsubscribe(ES_SHELL, SDLK_PAGEDOWN);

  this->completion = new ShellCompletion(this);
}

/**
 * standard deconstructor
*/
ShellInput::~ShellInput ()
{
  // delete what has to be deleted here
  delete this->completion;

  while (!this->history.empty())
  {
    this->history.pop_front();
  }
}

/**
 * sets the Repeate-delay and rate
 * @param repeatDelay the Delay it takes, to repeate a key
 * @param repeatRate the rate to repeate a pressed key
 */
void ShellInput::setRepeatDelay(float repeatDelay, float repeatRate)
{
  this->repeatDelay = repeatDelay;
  this->repeatRate = repeatRate;
}

/**
 * deletes the InputLine
 */
void ShellInput::flush()
{
  this->inputLine.clear();
  this->setText(this->inputLine);
}

/**
 * sets the entire text of the InputLine to text
 * @param text the new Text to set as InputLine
 */
void ShellInput::setInputText(const std::string& text)
{
  this->inputLine = text;
  this->setText(this->inputLine);
}


/**
 * adds one character to the inputLine
 * @param character the character to add to the inputLine
 */
void ShellInput::addCharacter(char character)
{
  if (this->historyScrolling)
  {
    this->history.pop_back();
    this->historyScrolling = false;
  }

  this->inputLine += character;
  this->setText(this->inputLine);
}

/**
 * adds multiple Characters to thr inputLine
 * @param characters a \\0 terminated char-array to add to the InputLine
 */
void ShellInput::addCharacters(const std::string& characters)
{
  if (this->historyScrolling)
  {
    this->history.pop_back();
    this->historyScrolling = false;
  }

  this->inputLine += characters;
  this->setText(this->inputLine);
}

/**
 * removes characterCount characters from the InputLine
 * @param characterCount the count of Characters to remove from the input Line
 */
void ShellInput::removeCharacters(unsigned int characterCount)
{
  if (this->historyScrolling)
  {
    this->history.pop_back();
    this->historyScrolling = false;
  }
  if (this->inputLine.size() < characterCount)
    characterCount = this->inputLine.size();

  this->inputLine.erase(this->inputLine.size() - characterCount, this->inputLine.size());
  this->setText(this->inputLine);
}

/**
 * executes the command stored in the inputLine
 * @return true if the command was commited successfully, false otherwise
 */
bool ShellInput::executeCommand()
{
  ShellBuffer::addBufferLineStatic("Execute Command: %s\n", this->inputLine.c_str());

  if (this->inputLine.empty())
    return false;

  ShellCommand::execute(this->inputLine);

  // removing the eventually added Entry (from scrolling) to the List
  if (this->historyScrolling)
  {
    this->history.pop_back();
    this->historyScrolling = false;
  }

  // adding the new Command to the History
  this->history.push_back(this->inputLine);
  if (this->history.size() > this->historyLength)
  {
    this->history.pop_front();
  }

  this->flush();

  return true;
}


/**
 * moves one entry up in the history.
 */
void ShellInput::historyMoveUp()
{
  if (!this->historyScrolling)
  {
    this->history.push_back(this->inputLine);
    this->historyScrolling = true;
    this->historyIT = --this->history.end();
  }

  if(this->historyIT != this->history.begin())
  {
    std::string prevElem = *(--this->historyIT);
    /*if (prevElem == NULL) /// TODO STD
      return;
    else */
    {
      this->flush();
      this->setInputText(prevElem);
    }
  }
}

/**
 * moves one entry down in the history
 */
void ShellInput::historyMoveDown()
{
  if (!this->historyScrolling)
    return;
  if (this->historyIT != this->history.end())
  {
    std::string nextElem = *(++this->historyIT);
    /*    if (nextElem == NULL) /// TODO FIX STD
      return;
    else */
    {
      this->flush();
      this->setInputText(nextElem);
    }
  }
}


/**
 * prints out some nice help about the Shell
 */
void ShellInput::help(const std::string& className, const std::string& functionName)
{
  printf("%s::%s\n", className.c_str(), functionName.c_str());

  if (className.empty())
  {
    PRINT(0)("Help for the most important Shell-commands\n");
    PRINT(0)("F1 - HELP; F2 - DEBUG; '`' - open/close shell\n");
    PRINT(0)("input order:\n");
    PRINT(0)("ClassName [objectName] function [parameter1, [parameter2 ...]]  or\n");
    PRINT(0)("Alias [parameter]\n");
    PRINT(0)("- Also try 'help className'");
  }
  else if (!className.empty() && functionName.empty())
  {
    ShellCommandClass::help(className);
    //PRINTF(1)("%s::%s\n", className, functionName);
  }
}

/**
 * ticks the ShellInput
 * @param dt the time passed since the last update
 */
void ShellInput::tick(float dt)
{
  if (this->delayed > 0.0)
    this->delayed -= dt;
  else if (this->pressedKey != SDLK_FIRST )
  {
    this->delayed = this->repeatRate;
    switch (this->pressedKey )
    {
      case SDLK_BACKSPACE:
        this->removeCharacters(1);
        break;
      case SDLK_UP:
        this->historyMoveUp();
        break;
      case SDLK_DOWN:
        this->historyMoveDown();
        break;
      default:
      {
        if (likely(pressedKey < 127))
          this->addCharacter(this->pressedKey);
      }
    }
  }
}

/**
 * listens for some event
 * @param event the Event happened
 */
void ShellInput::process(const Event &event)
{
  if (event.bPressed)
  {
    PRINTF(5)("Shell received command %s\n", SDLKToKeyname(event.type));
    if (event.type == SDLK_F1)
      this->help();
    else if (event.type == SDLK_F2)
    {
      ;//this->debug();
    }
    else if (event.type == SDLK_UP)
    {
      this->historyMoveUp();
      this->pressedKey = event.type;
    }
    else if (event.type == SDLK_DOWN)
    {
      this->historyMoveDown();
      this->pressedKey = event.type;
    }
    else if (event.type == SDLK_TAB)
      this->completion->autoComplete();
    else if (event.type == SDLK_BACKSPACE)
    {
      this->delayed = this->repeatDelay;
      this->pressedKey = SDLK_BACKSPACE;
      this->removeCharacters(1);
    }
    else if (event.type == SDLK_RETURN)
    {
      this->executeCommand();
      this->pressedKey = event.type;
    }
    // any other keyboard key
    else if (likely(event.type < 127))
    {
      this->addCharacter(event.x);
      this->pressedKey = event.x;
    }
    this->delayed = this->repeatDelay;
  }
  else // if(!event.bPressed)
  {
    if (this->pressedKey == event.x || this->pressedKey == event.type)
    {
      this->pressedKey = 0;
      this->delayed = 0.0;
    }
  }
}
