/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Christian Meyer
   co-programmer: Benjamin Grauer

   2005-06-10: some naming conventions

//
//  splitLine
//  STL string tokenizer
//
//  Created by Clemens Wacha.
//  Version 1.0
//  Copyright (c) 2005 Clemens Wacha. All rights reserved.
//

*/

#include "substring.h"

#include <string.h>
#include <cassert>

/**
 * @brief create a SubString from
 * @param string the String to Spilit
 * @param splitter the Character at which to split string (delimiter)
 */
SubString::SubString(const std::string& string, char splitter)
{
  char split[2];
  split[0] = splitter;
  split[1] = '\0';
  SubString::splitLine(this->strings, this->offsets,
                       string, split);
}

/**
 * @brief Splits a String into a SubString removing all whiteSpaces
 * @param string the String to Split
 * @param whiteSpaces MUST BE __TRUE__ or __FALSE__ (will be ignored)
 */
SubString::SubString(const std::string& string, bool whiteSpaces)
{
  SubString::splitLine(this->strings, this->offsets,
                       string);
}

/**
 * @brief Splits a String into multiple splitters.
 * @param string the String to split
 * @param splitters multiple set of characters at what to split. (delimiters)
 * @param escapeChar The Escape Character that overrides splitters commends and so on...
 * @param safemode_char within these characters splitting won't happen
 * @param comment_char the Comment character.
 */
SubString::SubString(const std::string& string, const std::string& splitters, char escapeChar,char safemode_char, char comment_char)
{
  SubString::splitLine(this->strings, this->offsets,
                       string, splitters, escapeChar, safemode_char, comment_char);
}

/**
 * @brief creates a SubSet of a SubString.
 * @param subString the SubString to take a set from.
 * @param subSetBegin the beginning to the end
 */
SubString::SubString(const SubString& subString, unsigned int subSetBegin)
{
  for (unsigned int i = subSetBegin; i < subString.size(); i++)
    this->strings.push_back(subString[i]);
}


/**
 * @brief creates a SubSet of a SubString.
 * @param subString the SubString to take a Set from
 * @param subSetBegin the beginning to the end
 * @param subSetEnd the end of the SubSet (max subString.size() will be checked internaly)
 */
SubString::SubString(const SubString& subString, unsigned int subSetBegin, unsigned int subSetEnd)
{
  for (unsigned int i = subSetBegin; i < subString.size() || i < subSetEnd; i++)
    this->strings.push_back(subString[i]);
}


/**
 * @brief removes the object from memory
 */
SubString::~SubString()
{ }

/**
 * @brief An empty String
 */
const std::string SubString::emptyString = "";

/**
 * @brief stores the Value of subString in this SubString
 * @param subString will be copied into this String.
 * @returns this SubString.
 */
SubString& SubString::operator=(const SubString& subString)
{
  this->offsets = subString.offsets;
  this->strings = subString.strings;
  return *this;
}


/**
 * @brief comparator.
 * @param subString the SubString to compare against this one.
 * @returns true if the Stored Strings match
 */
bool SubString::operator==(const SubString& subString)
{
  return (this->strings == subString.strings);
}


/**
 * @brief append operator
 * @param subString the String to append.
 * @returns a SubString where this and subString are appended.
 */
SubString SubString::operator+(const SubString& subString) const
{
  return SubString(subString) += subString;
}


/**
 * @brief append operator.
 * @param subString append subString to this SubString.
 * @returns this substring appended with subString
 */
SubString& SubString::operator+=(const SubString& subString)
{
  for (unsigned int i = 0; i < subString.size(); i++)
    this->strings.push_back(subString[i]);
}


/**
 * @brief Split the String at
 * @param string where to split
 * @param splitter delimiter.
 */
unsigned int SubString::split(const std::string& string, char splitter)
{
  this->offsets.clear();
  this->strings.clear();
  char split[2];
  split[0] = splitter;
  split[1] = '\0';
  SubString::splitLine(this->strings, this->offsets, string, split);
  return strings.size();
}


/**
 * @brief Splits a String into a Substring removing all whiteSpaces
 * @param string the String to Split
 * @param whiteSpaces MUST BE __TRUE__
 *
 */
unsigned int SubString::split(const std::string& string, bool whiteSpaces)
{
  this->offsets.clear();
  this->strings.clear();
  SubString::splitLine(this->strings, this->offsets, string);
  return strings.size();
}


/**
 * @brief Splits a String into multiple splitters.
 * @param string the String to split
 * @param splitters multiple set of characters at what to split. (delimiters)
 * @param escapeChar The Escape Character that overrides splitters commends and so on...
 * @param safemode_char within these characters splitting won't happen
 * @param comment_char the Comment character.
 */
unsigned int SubString::split(const std::string& string, const std::string& splitters, char escapeChar,char safemode_char, char comment_char)
{
  this->offsets.clear();
  this->strings.clear();
  SubString::splitLine(this->strings, this->offsets,
                       string, splitters, escapeChar, safemode_char);
  return strings.size();
}


/**
 * @brief joins together all Strings of this Substring.
 * @param delimiter the String between the subStrings.
 * @returns the joined String.
 */
std::string SubString::join(const std::string& delimiter) const
{
  if (!this->strings.empty())
  {
    std::string retVal = this->strings[0];
    for (unsigned int i = 0; i < this->strings.size(); i++)
      retVal += delimiter + this->strings[i];
    return retVal;
  }
  else
    return SubString::emptyString;
}


/**
 * @brief creates a SubSet of a SubString.
 * @param subSetBegin the beginning to the end
 * @returns the SubSet
 *
 * This function is added for your convenience, and does the same as
 * SubString::SubString(const SubString& subString, unsigned int subSetBegin)
 */
SubString SubString::getSubSet(unsigned int subSetBegin) const
{
  return SubString(*this, subSetBegin);
}


/**
 * @brief creates a SubSet of a SubString.
 * @param subSetBegin the beginning to
 * @param subSetEnd the end of the SubSet to select (if bigger than subString.size() it will be downset.)
 * @returns the SubSet
 *
 * This function is added for your convenience, and does the same as
 * SubString::SubString(const SubString& subString, unsigned int subSetBegin)
 */
SubString SubString::getSubSet(unsigned int subSetBegin, unsigned int subSetEnd) const
{
  return SubString(*this, subSetBegin, subSetEnd);
}


/**
 * @brief get a particular substring's offset
 * @param i the ID of the substring to get the offset from
 * @returns the offset or NULL if an invalid ID was given
 */
unsigned int SubString::getOffset(unsigned int i) const
{
  if( i < this->offsets.size() && i >= 0)
    return this->offsets[i];
  else
    return 0;
}


/**
 * @brief splits line into tokens and stores them in ret.
 * @param ret the Array, where the Splitted strings will be stored in
 * @param offsets an Array of Offsets, here the distance from the inputstring
 * to the beginning of the current token is stored
 * @param line the inputLine to split
 * @param delimiters a String of Delimiters (here the input will be splitted)
 * @param escape_char: Escape carater (escapes splitters)
 * @param safemode_char: the beginning of the safemode is marked with this
 * @param comment_char: the beginning of a comment is marked with this: (until the end of a Line)
 * @param start_state: the Initial state on how to parse the String.
 * @returns SPLIT_LINE_STATE the parser was in when returning
 *
 * This is the Actual Splitting Algorithm from Clemens Wacha
 * Supports delimiters, escape characters,
 * ignores special  characters between safemode_char and between comment_char and linend '\n'.
 *
 */
SPLIT_LINE_STATE SubString::splitLine(std::vector<std::string>& ret, std::vector<unsigned int>& offsets,
                                      const std::string& line, const std::string& delimiters,
                                      char escape_char, char safemode_char, char comment_char,
                                      SPLIT_LINE_STATE start_state)
{
  SPLIT_LINE_STATE state = start_state;
  unsigned int i = 0;
  std::string token;

  if(start_state != SL_NORMAL && ret.size() > 0)
  {
    token = ret[ret.size()-1];
    ret.pop_back();
  }

  while(i < line.size())
  {
    switch(state)
    {
      case SL_NORMAL:
        if(line[i] == escape_char)
        {
          state = SL_ESCAPE;
        }
        else if(line[i] == safemode_char)
        {
          state = SL_SAFEMODE;
        }
        else if(line[i] == comment_char)
        {
          /// FINISH
          if(token.size() > 0)
          {
            ret.push_back(token);
            offsets.push_back(i);
            token.clear();
          }
          token += line[i];       // EAT
          state = SL_COMMENT;
        }
        else if(delimiters.find(line[i]) != std::string::npos)
        {
          // line[i] is a delimiter
          /// FINISH
          if(token.size() > 0)
          {
            ret.push_back(token);
            offsets.push_back(i);
            token.clear();
          }
        }
        else
        {
          token += line[i];       // EAT
        }
        break;
      case SL_ESCAPE:
        if(line[i] == 'n') token += '\n';
        else if(line[i] == 't') token += '\t';
        else if(line[i] == 'v') token += '\v';
        else if(line[i] == 'b') token += '\b';
        else if(line[i] == 'r') token += '\r';
        else if(line[i] == 'f') token += '\f';
        else if(line[i] == 'a') token += '\a';
        else if(line[i] == '?') token += '\?';
        else token += line[i];  // EAT
        state = SL_NORMAL;
        break;
      case SL_SAFEMODE:
        if(line[i] == safemode_char)
        {
          state = SL_NORMAL;
        }
        else if(line[i] == escape_char)
        {
          state = SL_SAFEESCAPE;
        }
        else
        {
          token += line[i];       // EAT
        }
        break;
      case SL_SAFEESCAPE:
        if(line[i] == 'n') token += '\n';
        else if(line[i] == 't') token += '\t';
        else if(line[i] == 'v') token += '\v';
        else if(line[i] == 'b') token += '\b';
        else if(line[i] == 'r') token += '\r';
        else if(line[i] == 'f') token += '\f';
        else if(line[i] == 'a') token += '\a';
        else if(line[i] == '?') token += '\?';
        else token += line[i];  // EAT
        state = SL_SAFEMODE;
        break;
      case SL_COMMENT:
        if(line[i] == '\n')
        {
          /// FINISH
          if(token.size() > 0)
          {
            ret.push_back(token);
            offsets.push_back(i);
            token.clear();
          }
          state = SL_NORMAL;
        }
        else
        {
          token += line[i];       // EAT
        }
        break;
      default:
        // nothing
        break;
    }
    i++;
  }

  /// FINISH
  if(token.size() > 0)
  {
    ret.push_back(token);
    offsets.push_back(i);
    token.clear();
  }
  return(state);
}


/**
 * @brief Some nice debug information about this SubString
 */
void SubString::debug() const
{
  printf("Substring-information::count=%d ::", this->strings.size());
  for (unsigned int i = 0; i < this->strings.size(); i++)
    printf("s%d='%s'::", i, this->strings[i].c_str());
  printf("\n");
}
