/*!
 * @file ogg_player.h
 * Ogg-Player definition
 */


#ifndef _OGG_PLAYER_H
#define _OGG_PLAYER_H

using namespace std;

#include "base_object.h"

#include "alincl.h"
#include <ogg/ogg.h>
#include <vorbis/vorbisfile.h>
#include <SDL_thread.h>

struct File;


#define OGG_PLAYER_BUFFER_SIZE (8096 * 2)


// the definition of a Ogg-Player
class OggPlayer : public BaseObject
{
public:
  /**
   * An enumerator defining in what State the OggPlayer is.
   * The OggPlayer can be in multiple States listed here.
   * It can as an example be in FileOpened and Stopped.
   */
  typedef enum {
    None                   = 0x000,   //!< Initialized
    FileOpened             = 0x100,   //!< File is Opened
    SourceAllocated        = 0x200,   //!< Source is Allocated.
    BuffersAllocated       = 0x400,   //!< 2 Buffers are Allocated.
    Stopped                = 0x010,   //!< OggPlayer is stopped.
    Playing                = 0x020,   //!< OggPlayer is Playing.
    Paused                 = 0x040,   //!< OggPlayer is Paused.
    Error                  = 0x001,   //!< An Error has occured.
  } State;

  OggPlayer(const std::string& fileName = "");
  virtual ~OggPlayer();
  /** @param state compare this State with the internal State @returns true on match */
  bool operator==(OggPlayer::State state) { return this->state & state; };

  bool open(const std::string& fileName = "");

  bool play();
  void stop();
  void pause();
  void rewind(); // convenience
  void jumpTo(float timeCode);

  float length();
  bool isPlaying();
  bool getState() { return this->state; };

  void debug() const;
  void printState() const;
  const char* getVorbisError(int code);


private:
  static int musicThread(void* oggPlayer);
  bool playback();
  void suspend();
  bool update();

  void release();

  bool stream(ALuint buffer);
  void empty();

private:
  FILE*               oggFile;              //!< general file-handler, to open the sound-file
  OggVorbis_File      oggStream;            //!< The stream this Ogg-player is playing back
  vorbis_info*        vorbisInfo;           //!< The information held in the opened ogg-file
  vorbis_comment*     vorbisComment;        //!< Comments about the ogg-file

  ALuint              buffers[2];           //!< buffers that handle sequentially buffering of the audio
  ALuint              source;               //!< The source we play back on
  ALenum              format;               //!< The format we play back
  unsigned int        state;                //!< The States the OggPlayer is in (this can be multiple entries from OggPlayer::State).

  SDL_Thread*         musicThreadID;        //!< The Thread in which music is Played back.
  SDL_mutex*          musicMutex;           //!< A Mutex so that the two threads do not interfere.

  class MutexLock
  {
  public:
    MutexLock(SDL_mutex* mutex) { SDL_mutexP(mutex); this->mutex = mutex; };
    ~MutexLock() { SDL_mutexV(mutex); };
  private:
    SDL_mutex* mutex;
  };
};


#endif /* _OGG_PLAYER_H */
