/*!
 * @file event_handler.h
 * Definition of the EventHandler
 *
 */

#ifndef _EVENT_HANDLER_H
#define _EVENT_HANDLER_H

#include "base_object.h"
#include "key_mapper.h"
#include "event_def.h"
#include <stack>

// FORWARD DECLARATION
class EventListener;

//! The one Event Handler from Orxonox
class EventHandler : public BaseObject {

 public:
  virtual ~EventHandler();
  /** @returns a Pointer to the only object of this Class */
  inline static EventHandler* getInstance() { if (!singletonRef) singletonRef = new EventHandler();  return singletonRef; };
  void init();

  /** @param state: to which the event handler shall change */
  inline void setState(elState state) { this->state = state; };
  /** @returns the current state */
  inline elState getState() const { return this->state; };

  void pushState(elState state);
  elState popState();

  void subscribe(EventListener* el, elState state, int eventType);
  void unsubscribe(elState state, int eventType);
  void unsubscribe(EventListener* el, elState state = ES_ALL);
  void flush(elState state);
  /** @returns true, if the @param state has @param eventType subscribed?? */
  inline bool isSubscribed(elState state, int eventType) { return(listeners[state][eventType] == NULL)?false:true; };


  void withUNICODE(bool enableUNICODE);
  void grabEvents(bool grabEvents);
  bool grabbedEvents() const { return this->eventsGrabbed; };

  void process();

  static int eventFilter(const SDL_Event *event);
  void debug() const;

 private:
  EventHandler();

 private:
  static EventHandler*       singletonRef;                    //!< the singleton reference

  EventListener*             listeners[ES_NUMBER][EV_NUMBER]; //!< a list of registered listeners.
  elState                    state;                           //!< the state of the event handlder.
  std::stack<short>          stateStack;                      //!< a stack for the States we are in.
  KeyMapper*                 keyMapper;                       //!< reference to the key mapper.

  bool                       bUNICODE;                        //!< If unicode should be enabled.
  bool                       eventsGrabbed;                   //!< If the events should be grabbed
};


#endif /* _EVENT_HANDLER_H */
