/*!
 * @file texture.h
 * @brief Contains the texture class, that handles the reading of Images into Texutre-files.
 */

#ifndef _TEXTURE_H
#define _TEXTURE_H

#include "base_object.h"
#include "lib/util/loading/resource.h"

#include "glincl.h"

/* Forward Declaration */
struct SDL_Surface;

//! A Class, that reads in Textures from different fileformats.
  class Texture : public Loading::Resource
  {
    public:
      Texture(const std::string& imageName = "", GLenum target = GL_TEXTURE_2D);
  //  Texture(TEXTURE_TYPE type, int resolution);
      virtual ~Texture();

      bool loadImage(const std::string& imageName, GLenum target = GL_TEXTURE_2D);
      virtual bool rebuild();

      /** @returns The textureID of this texture.  */
      inline GLuint getTexture() const { return this->texture; };
      /** @returns true if texture has alpha, false otherwise */
      inline bool hasAlpha() const {return bAlpha;}
      /** @returns the stored image of this Texture */
      const SDL_Surface* const getStoredImage() const { return this->image; };



      static void setTextureEnableState(bool texturesEnabled);
      /** @returns true if Textures are enabled */
      inline static bool getTextureEnableState() { return Texture::texturesEnabled; };

      // Utility functionality:
      SDL_Surface* prepareSurface(SDL_Surface* input, bool& hasAlpha) const;
      GLuint loadTexToGL (const SDL_Surface* surface, GLenum target = GL_TEXTURE_2D) const;

    protected:
      bool setSurface(SDL_Surface* newSurface);
      bool setAlpha(bool hasAlpha) { this->bAlpha = hasAlpha; };
      bool setTexture(GLuint texture) { this->texture = texture; };


    private:
      GLuint           texture;            //!< The Texture-ID of opengl from this Texture.
      bool             bAlpha;             //!< if the texture has an alpha channel.
      SDL_Surface*     image;              //!< The SDL_Surfce that stores the Texture on it.
      GLclampf         priority;           //!< the priority of the current texture (used for garphics cards with limited mem)

      static bool      texturesEnabled;    //!< If the Textures are enabled.
  };

#endif /* _TEXTURE_H */
