/*!
 * @file shell_command.h
 * Definition of a on-screen-shell
 */

#ifndef _SHELL_COMMAND_H
#define _SHELL_COMMAND_H

#include "base_object.h"

#include "executor/executor.h"
#include <stdarg.h>

#define     SHELL_COMMAND_MAX_SIZE      //!< The maximum size of a Shell Command



// FORWARD DECLARATION
class ShellCommandClass;
class ShellCommandAlias;

/**
 * an easy to use Macro to create a Command
 * @param command the name of the command (without "" around the string)
 * @param class the name of the class to apply this command to (without the "" around the string)
 * @param function the function to call
 *
 * MEANING:
 *  ShellCommand* someUniqueVarName =
 *       ShellCommand<ClassName>::registerCommand("commandNameInShell", "ClassName", &ClassName::FunctionToCall);
 *
 * In the Shell you would call this Command using:
 * $ ClassName [ObjectName] commandNameInShell [parameters]
 */
//#define SHELL_COMMAND(command, class, function) \
//        ShellCommand* shell_command_##class##_##command = ShellCommand<class>::registerCommand(#command, #class, &class::function)
#define SHELL_COMMAND(command, class, function) \
           ShellCommand* shell_command_##class##_##command = ShellCommand::registerCommand(#command, #class, ExecutorObjective<class>(&class::function))

/**
 * an easy to use Macro to create a Command
 * @param command the name of the command (without "" around the string)
 * @param class the name of the class to apply this command to (without the "" around the string)
 * @param function the function to call
 *
 * MEANING:
 *  ShellCommand* someUniqueVarName =
 *       ShellCommand<ClassName>::registerCommand("commandNameInShell", "ClassName", &ClassName::FunctionToCall);
 *
 * In the Shell you would call this Command using:
 * $ ClassName [ObjectName] commandNameInShell [parameters]
 */
#define SHELL_COMMAND_STATIC(command, class, function) \
           ShellCommand* shell_command_##class##_##command = ShellCommand::registerCommand(#command, #class, ExecutorStatic<class>(function))



//! a baseClass for all possible ShellCommands
class ShellCommand : public BaseObject
{
  friend class ShellCommandClass;
  public:
    static bool execute (const std::string& executionString);

    ShellCommand* describe(const std::string& description);
    ShellCommand* setAlias(const std::string& alias);
    ShellCommand* defaultValues(const MultiType& value0 = MT_NULL, const MultiType& value1 = MT_NULL,
                                const MultiType& value2 = MT_NULL, const MultiType& value3 = MT_NULL,
                                const MultiType& value4 = MT_NULL);

    static ShellCommand* registerCommand(const std::string& commandName, const std::string& className, const Executor& executor);

    static void unregisterCommand(const std::string& commandName, const std::string& className);

    static void debug();

  protected:
    ShellCommand(const std::string& commandName, const std::string& className, const Executor& executor);
    ~ShellCommand();

    static bool isRegistered(const std::string& commandName, const std::string& className);
    static const char* paramToString(long parameter);

  protected:
    MultiType*                       defaultValue;                         //!< Default Values.

  private:
    ShellCommandClass*               shellClass;                           //!< A Pointer to the Shell-Class this Command belongs to.
    ShellCommandAlias*               alias;                                //!< An Alias for the Class.

    std::string                      description;                          //!< A description for this commnand. (initially NULL). Assigned with (create)->describe("blablabla");
    Executor*                        executor;                             //!< The Executor, that really executes the Function.

};

//! A Class, that handles aliases.
class ShellCommandAlias
{
  friend class ShellCommand;
  public:
    /** @returns the Name of the Alias. */
    const std::string& getName() const { return this->aliasName; };
    /** @returns the Command, this Alias is asociated with */
    ShellCommand* getCommand() const { return this->command; };

  private:
    /** @param aliasName the name of the Alias @param command the Command, to associate this alias with */
    ShellCommandAlias(const std::string& aliasName, ShellCommand* command) { this->aliasName = aliasName; this->command = command; };

  private:
    std::string     aliasName;       //!< the name of the Alias
    ShellCommand*   command;         //!< a pointer to the command, this alias executes.
};

#endif /* _SHELL_COMMAND_H */
