/*!
 * @file static_model.h
 * @brief Contains the Model Class that handles Static 3D-Models rendered with glList's
 */

#ifndef _STATIC_MODEL_H
#define _STATIC_MODEL_H

#include "model.h"

#include "material.h"
#include "glincl.h"
#include <vector>
#include <list>

// definition of different modes for setting up Faces
#define VERTEX 0       //!< If Faces are created WITH Vertex-Coordinate
#define NORMAL 1       //!< If Faces are created WITH Normals (otherwise autocalculate)
#define TEXCOORD 2     //!< If Faces are created WITH TextureCoordinate

//! an enumerator for VERTEX_FORMAT
typedef enum VERTEX_FORMAT {
  VERTEX_ONLY = VERTEX,
  VERTEX_NORMAL = NORMAL,
  VERTEX_TEXCOORD = TEXCOORD,
  VERTEX_TEXCOORD_NORMAL = NORMAL | TEXCOORD
};

////////////////////
/// SUB-ELEMENTS ///
////////////////////
//! This is the placeholder of one Vertex beloning to a Face.
class ModelFaceElement
{
 public:
  ModelFaceElement();
  ~ModelFaceElement();

  int                 vertexNumber;         //!< The number of the Vertex out of the Array* vertices, this vertex points to.
  int                 normalNumber;         //!< The number of the Normal out of the Array* normals, this vertex points to.
  int                 texCoordNumber;       //!< The number of the textureCoordinate out of the Array* vTexture, this vertex points to.

  ModelFaceElement*   next;                 //!< Point to the next FaceElement in this List.
};

//! This is the placeholder of a Face belonging to a Group of Faces.
class ModelFace
{
 public:
  ModelFace();
  ~ModelFace();

  unsigned int        vertexCount;     //!< The Count of vertices this Face has.
  ModelFaceElement*   firstElem;       //!< Points to the first Vertex (FaceElement) of this Face.
  Material*           material;        //!< The Material to use.

  ModelFace*          next;            //!< Pointer to the next Face.
};

//! Group to handle multiple Models per obj-file.
class ModelGroup
{
 public:
  ModelGroup();
  ~ModelGroup();

  void cleanup();

  std::string  name;           //!< the Name of the Group. this is an identifier, that can be accessed via the draw (std::string name) function.
  GLubyte*     indices;        //!< The indices of the Groups. Needed for vertex-arrays
  GLuint       listNumber;     //!< The number of the GL-List this Group gets.
  ModelFace*   firstFace;      //!< The first Face in this group.
  ModelFace*   currentFace;    //!< The current Face in this Group (the one we are currently working with.)
  int          faceMode;       //!< The Mode the Face is in: initially -1, 0 for FaceList opened, 1 for Material,  3 for triangle, 4 for Quad, 5+ for Poly @todo ENUM...
  int          faceCount;      //!< The Number of Faces this Group holds.

  ModelGroup*  next;           //!< Pointer to the next Group.
};

struct ModelMaterial
{
  Material* material;
  bool external;
};

/////////////
/// MODEL ///
/////////////
//! Class that handles static 3D-Models.
/**
 * it can also read them in and display them.
 * All the objects are rendered with glLists
 */
class StaticModel : public Model
{
 public:
  StaticModel(const std::string& modelName = "");
  virtual ~StaticModel();

  virtual void draw() const;
  void draw(int groupNumber) const;
  void draw(const std::string& groupName) const;

  void rebuild();

  Material* addMaterial(Material* material);
  Material* addMaterial(const std::string& materialName);

  bool addGroup(const std::string& groupString);

  bool addVertex(const std::string& vertexString);
  bool addVertex(float x, float y, float z);

  bool addFace(const std::string& faceString);
  bool addFace(int faceElemCount, VERTEX_FORMAT type, ...);

  bool addVertexNormal(const std::string& normalString);
  bool addVertexNormal(float x, float y, float z);

  bool addVertexTexture(const std::string& vTextureString);
  bool addVertexTexture(float u, float v);

  bool setMaterial(const std::string& mtlString);
  bool setMaterial(Material* mtl);

  void finalize();


 protected:
  void cubeModel();

  Material* findMaterialByName(const std::string& materialName);

 protected:
  float            scaleFactor;     //!< The Factor with which the Model should be scaled. @todo maybe one wants to scale the Model after Initialisation

 private:
  bool buildVertexNormals();

  bool importToDisplayList();
  bool buildTriangleList();

  bool addGLElement(ModelFaceElement* elem);

  bool cleanup();

 private:
  bool                       finalized;       //!< Sets the Object to be finalized.

  unsigned int               faceCount;       //!< A modelwide Counter for the faces

  std::vector<GLfloat>       vertices;        //!< The Array that handles the Vertices.
  std::vector<GLfloat>       normals;         //!< The Array that handles the Normals.
  std::vector<GLfloat>       vTexture;        //!< The Array that handles the VertexTextureCoordinates.

  ModelGroup*                firstGroup;      //!< The first of all groups.
  ModelGroup*                currentGroup;    //!< The currentGroup. this is the one we will work with.
  int                        groupCount;      //!< The Count of Groups.

  std::list<ModelMaterial*>  materialList;    //!< A list for all the Materials in this Model
};

#endif
