/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_IMPORTER

#include "md2Model.h"
#include "material.h"

#include "debug.h"
#include "util/loading/resource_manager.h"


using namespace std;

//! the model anorms
sVec3D MD2Model::anorms[NUM_VERTEX_NORMALS] = {
 #include "anorms.h"
};

//! anormal dots, no idea of how this shall work, but it does
float MD2Model::anormsDots[SHADEDOT_QUANT][256] = {
  #include "anormtab.h"
};


//! the angle under which the model is viewd, used internaly
float md2Angle = 0.0f;


//! list of all different animations a std md2model supports
sAnim MD2Model::animationList[21] =
  {
 // begin, end, fps, interruptable
    {   0,  39,  9, 1 },   //!< STAND
    {  40,  45, 10, 1 },   //!< RUN
    {  46,  53, 10, 0 },   //!< ATTACK
    {  54,  57,  7, 1 },   //!< PAIN_A
    {  58,  61,  7, 1 },   //!< PAIN_B
    {  62,  65,  7, 1 },   //!< PAIN_C
    {  66,  71,  7, 0 },   //!< JUMP
    {  72,  83,  7, 1 },   //!< FLIP
    {  84,  94,  7, 1 },   //!< SALUTE
    {  95, 111, 10, 1 },   //!< FALLBACK
    { 112, 122,  7, 1 },   //!< WAVE
    { 123, 134,  6, 1 },   //!< POINTT
    { 135, 153, 10, 1 },   //!< CROUCH_STAND
    { 154, 159,  7, 1 },   //!< CROUCH_WALK
    { 160, 168, 10, 1 },   //!< CROUCH_ATTACK
    { 196, 172,  7, 1 },   //!< CROUCH_PAIN
    { 173, 177,  5, 0 },   //!< CROUCH_DEATH
    { 178, 183,  10, 0 },   //!< DEATH_FALLBACK
    { 184, 189,  10, 0 },   //!< DEATH_FALLFORWARD
    { 190, 197,  10, 0 },   //!< DEATH_FALLBACKSLOW
    { 198, 198,  5, 1 },   //!< BOOM
  };



/********************************************************************************
 *   MD2Model                                                                   *
 ********************************************************************************/

/**
  \brief simple constructor initializing all variables
*/
MD2Model::MD2Model(const std::string& modelFileName, const std::string& skinFileName, float scale)
{
  this->setClassID(CL_MD2_MODEL, "MD2Model");
  /* this creates the data container via ressource manager */
  if (!modelFileName.empty())
    this->data = (MD2Data*)ResourceManager::getInstance()->load(modelFileName, MD2, RP_GAME, skinFileName, scale);
  if( unlikely(this->data == NULL))
    PRINTF(0)("The model was not found, MD2Model Loader finished abnormaly. Update the data-repos\n");

  this->scaleFactor = scale;

  shadeDots = MD2Model::anormsDots[0];
  /* set the animation stat mannualy */
  this->animationState.type = STAND;
  this->animationState.numPlays = 1;
  this->setAnim(STAND);

  this->debug();

    //write the modelinfo information
  this->pModelInfo.numVertices = this->data->numVertices;
  this->pModelInfo.numTriangles = this->data->numTriangles;
  this->pModelInfo.numNormals = 0;
  this->pModelInfo.numTexCoor = this->data->numTexCoor;
  this->pModelInfo.pVertices = (float*)this->data->pVertices;
  this->pModelInfo.pNormals = NULL;
  this->pModelInfo.pTexCoor = (float*)this->data->pTexCoor;

  // triangle conversion
  this->pModelInfo.pTriangles = new sTriangleExt[this->data->numTriangles];
  for( int i = 0; i < this->data->numTriangles; i++)
  {
    this->pModelInfo.pTriangles[i].indexToVertices[0] = this->data->pTriangles[i].indexToVertices[0];
    this->pModelInfo.pTriangles[i].indexToVertices[1] = this->data->pTriangles[i].indexToVertices[1];
    this->pModelInfo.pTriangles[i].indexToVertices[2] = this->data->pTriangles[i].indexToVertices[2];

    this->pModelInfo.pTriangles[i].indexToTexCoor[0] = this->data->pTriangles[i].indexToTexCoor[0];
    this->pModelInfo.pTriangles[i].indexToTexCoor[1] = this->data->pTriangles[i].indexToTexCoor[1];
    this->pModelInfo.pTriangles[i].indexToTexCoor[2] = this->data->pTriangles[i].indexToTexCoor[2];
  }
}


/**
  \brief simple destructor, dereferencing all variables

  this is where the ressource manager is cleaning the stuff
*/
MD2Model::~MD2Model()
{
  ResourceManager::getInstance()->unload(this->data);
}


/**
 *  initializes an array of vert with the current frame scaled vertices
 * @param this->verticesList: the list of vertices to interpolate between

   we won't use the pVertices array directly, since its much easier and we need
   saving of data anyway
*/
void MD2Model::interpolate(/*sVec3D* this->verticesList*/)
{
  sVec3D* currVec;
  sVec3D* nextVec;

  currVec = &this->data->pVertices[this->data->numVertices * this->animationState.currentFrame];
  nextVec = &this->data->pVertices[this->data->numVertices * this->animationState.nextFrame];

  for( int i = 0; i < this->data->numVertices; ++i)
    {
      this->verticesList[i][0] = currVec[i][0] + this->animationState.interpolationState * (nextVec[i][0] - currVec[i][0]);
      this->verticesList[i][1] = currVec[i][1] + this->animationState.interpolationState * (nextVec[i][1] - currVec[i][1]);
      this->verticesList[i][2] = currVec[i][2] + this->animationState.interpolationState * (nextVec[i][2] - currVec[i][2]);
    }
}


/**
  \brief sets the animation type
* @param type: animation type

  the animation types can be looked up in the animationType table
*/
void MD2Model::setAnim(int type, int animPlayback)
{
  if( (type < 0) || (type > MAX_ANIMATIONS) )
    type = STAND;

  if( MD2Model::animationList[this->animationState.type].bStoppable == 0)
  {
    if( this->animationState.numPlays == 0 )
      return;
  }

  this->animationState.startFrame = animationList[type].firstFrame;
  this->animationState.endFrame = animationList[type].lastFrame;
  this->animationState.nextFrame = animationList[type].firstFrame + 1;
  this->animationState.fps = animationList[type].fps;
  this->animationState.type = type;
  this->animationState.numPlays = 0;
  this->animationState.animPlaybackMode = animPlayback;

  this->animationState.interpolationState = 0.0f;
  this->animationState.localTime = 0.0f;
  this->animationState.lastTime = 0.0f;
  this->animationState.currentFrame = animationList[type].firstFrame;
}


/**
  \brief sets the time in seconds passed since the last tick
* @param time: in sec
*/
void MD2Model::tick(float time)
{
  this->animate(time);
  this->processLighting();
  this->interpolate(/*this->verticesList*/);
}


/**
 * @brief draws the model: interface for all other classes out in the world
 * @todo make it const and virtual
 * FIXME
 */
void MD2Model::draw() const
{
  glPushMatrix();
  this->renderFrame();
  // renderFrameTriangles();
  glPopMatrix();
}


/**
  \brief this is an internal function to render this special frame selected by animate()
*/
void MD2Model::renderFrame() const
{
  int* pCommands = this->data->pGLCommands;

  /* some face culling stuff */
  glPushAttrib(GL_POLYGON_BIT);
  glFrontFace(GL_CW);
  glEnable(GL_CULL_FACE);
  glCullFace(GL_BACK);

  this->data->material.select();

  /* draw the triangles */
  while( int i = *(pCommands++)) /* strange looking while loop for maximum performance */
    {
      if( i < 0)
        {
          glBegin(GL_TRIANGLE_FAN);
          i = -i;
        }
      else
        {
          glBegin(GL_TRIANGLE_STRIP);
        }

      for(; i > 0; i--, pCommands += 3) /* down counting for loop, next 3 gl commands */
        {
          glTexCoord2f( ((float *)pCommands)[0], ((float *)pCommands)[1] );
          glNormal3fv(anorms[this->data->pLightNormals[pCommands[2]]]);
          glVertex3fv(this->verticesList[pCommands[2]]);
        }
      glEnd();

    }
  glDisable(GL_CULL_FACE);
  glPopAttrib();
}


void MD2Model::renderFrameTriangles() const
{
  //static sVec3D this->verticesList[MD2_MAX_VERTICES]; /* performance: created only once in a lifetime */
  int* pCommands = this->data->pGLCommands;
  /* some face culling stuff */
//   glPushAttrib(GL_POLYGON_BIT);
//   glFrontFace(GL_CW);
//   glEnable(GL_CULL_FACE);
//   glCullFace(GL_BACK);
//
//   this->processLighting();
//   this->interpolate(/*this->verticesList*/);
  this->data->material.select();

  /* draw the triangles */
  glBegin(GL_TRIANGLES);

  for( int i = 0, k = 0; i < this->data->numTriangles; ++i, k += 3)
  {
    float* v = this->data->pVertices[this->data->pTriangles[i].indexToVertices[0]];

    printf("triangle: %i\n", i);
    printf("     v0: (%f, %f, %f)\n", v[0], v[1], v[2]);
    v = this->data->pVertices[this->data->pTriangles[i].indexToVertices[1]];
    printf("     v1: (%f, %f, %f)\n", v[0], v[1], v[2]);
    v = this->data->pVertices[this->data->pTriangles[i].indexToVertices[2]];
    printf("     v2: (%f, %f, %f)\n", v[0], v[1], v[2]);


    glNormal3f(anorms[i][0], anorms[i][1], anorms[i][2]);
    glVertex3fv(this->data->pVertices[this->data->pTriangles[i].indexToVertices[0]]);

    glNormal3f(anorms[i][0], anorms[i][1], anorms[i][2]);
    glVertex3fv(this->data->pVertices[this->data->pTriangles[i].indexToVertices[1]]);

    glNormal3f(anorms[i][0], anorms[i][1], anorms[i][2]);
    glVertex3fv(this->data->pVertices[this->data->pTriangles[i].indexToVertices[2]]);
  }

  glEnd();
}


/**
  \brief animates the current model

  depending on the time passed (tick function), the player will select another model
*/
void MD2Model::animate(float time)
{
  this->animationState.localTime += time;

  if( this->animationState.localTime - this->animationState.lastTime > (1.0f / this->animationState.fps))
    {
      this->animationState.currentFrame = this->animationState.nextFrame;
      this->animationState.nextFrame++;

      if( this->animationState.nextFrame > this->animationState.endFrame )
      {
        if( this->animationState.animPlaybackMode == MD2_ANIM_LOOP)
        {
          this->animationState.nextFrame = this->animationState.startFrame;
          this->animationState.numPlays++;
        }
        else
        {
          this->animationState.nextFrame = this->animationState.endFrame;
        }
      }
      this->animationState.lastTime = this->animationState.localTime;
    }

//     if( this->animationState.currentFrame > (this->data->numFrames - 1) )
//       this->animationState.currentFrame = 0;

//     if( (this->animationState.nextFrame > (this->data->numFrames - 1)) && this->animationState.animPlaybackMode == MD2_ANIM_LOOP)
//     this->animationState.nextFrame = 0;

  this->animationState.interpolationState = this->animationState.fps *
    (this->animationState.localTime - this->animationState.lastTime);
}


/**
  \brief this is how id is precessing their lightning

  the details of how the whole lighting process is beeing handled - i have no idea... :)
*/
void MD2Model::processLighting()
{
  shadeDots = anormsDots[((int)(md2Angle*(SHADEDOT_QUANT / 360.0)))&(SHADEDOT_QUANT - 1)];
}


/**
  \brief prints out debug informations
*/
void MD2Model::debug()
{
  PRINT(0)("\n==========================| MD2Model::debug() |===\n");
  PRINT(0)("=  Model FileName:\t%s\n", this->data->fileName.c_str());
  PRINT(0)("=  Skin FileName:\t%s\n", this->data->skinFileName.c_str());
  PRINT(0)("=  Size in Memory:\t%i Bytes\n", this->data->header->frameSize * this->data->header->numFrames + 64); // 64bytes is the header size
  PRINT(0)("=  Number of Vertices:\t%i\n", this->data->header->numVertices);
  PRINT(0)("=  Number of Frames: \t%i\n", this->data->header->numFrames);
  PRINT(0)("=  Height, Width:\t%i, %i\n", this->data->header->skinHeight, this->data->header->skinWidth);
  PRINT(0)("=  Pointer to the data object: %p\n", this->data);
  PRINT(0)("===================================================\n\n");
}


/********************************************************************************
 *   MD2Data                                                                    *
 ********************************************************************************/

/**
  \brief simple constructor
*/
MD2Data::MD2Data(const std::string& modelFileName, const std::string& skinFileName, float scale)
{
  scale *= 0.1f;

  this->pVertices = NULL;
  this->pGLCommands = NULL;
  this->pLightNormals = NULL;
  this->pTexCoor = NULL;

  this->numFrames = 0;
  this->numVertices = 0;
  this->numGLCommands = 0;
  this->numTexCoor = 0;

//   this->scaleFactor = 1.0f;
  this->scaleFactor = scale;

  this->fileName = "";
  this->skinFileName = "";
  this->loadModel(modelFileName);
  this->loadSkin(skinFileName);
}


/**
  \brief simple destructor

  this will clean out all the necessary data for a specific md2model
*/
MD2Data::~MD2Data()
{
  delete this->header;

  delete [] this->pVertices;
  delete [] this->pGLCommands;
  delete [] this->pLightNormals;
  delete [] this->pTexCoor;

}



/**
  \brief this will load the whole model data (vertices, opengl command list, ...)
* @param fileName: the name of the model file
  \return true if success
*/
bool MD2Data::loadModel(const std::string& fileName)
{
  FILE *pFile;                            //file stream
  char* buffer;                           //buffer for frame data
  sFrame* frame;                          //temp frame
  sVec3D *pVertex;
  int* pNormals;

  //! @todo this chek should include deleting a loaded model (eventually)
  if (fileName.empty())
    return false;

  pFile = fopen(fileName.c_str(), "rb");
  if( unlikely(!pFile))
    {
      PRINTF(1)("Couldn't open the MD2 File for loading. Exiting.\n");
      return false;
    }
  this->header = new MD2Header;
  fread(this->header, 1, sizeof(MD2Header), pFile);
  /* check for the header version: make sure its a md2 file :) */
  if( unlikely(this->header->version != MD2_VERSION) && unlikely(this->header->ident != MD2_IDENT))
    {
      PRINTF(1)("Couldn't load file %s: invalid file format: stop loading\n", fileName.c_str());
      return false;
    }

  this->fileName =fileName;
  /* got the data: map it to locals */
  this->numFrames = this->header->numFrames;
  this->numVertices = this->header->numVertices;
  this->numTriangles = this->header->numTriangles;
  this->numGLCommands = this->header->numGlCommands;
  this->numTexCoor = this->header->numTexCoords;
  /* allocate memory for the data storage */
  this->pVertices = new sVec3D[this->numVertices * this->numFrames];
  this->pGLCommands = new int[this->numGLCommands];
  this->pLightNormals = new int[this->numVertices * this->numFrames];
  this->pTriangles = new sTriangle[this->numTriangles];
  this->pTexCoor = new sTexCoor[this->numTexCoor];
  buffer = new char[this->numFrames * this->header->frameSize];


  /* read frame data from the file to a temp buffer */
  fseek(pFile, this->header->offsetFrames, SEEK_SET);
  fread(buffer, this->header->frameSize, this->numFrames, pFile);
  /* read opengl commands */
  fseek(pFile, this->header->offsetGlCommands, SEEK_SET);
  fread(this->pGLCommands, sizeof(int), this->numGLCommands, pFile);
  /* triangle list */
  fseek(pFile, this->header->offsetTriangles, SEEK_SET);
  fread(this->pTriangles, sizeof(sTriangle), this->numTriangles, pFile);
  /*  read in texture coordinates */
  fseek(pFile, this->header->offsetTexCoords, SEEK_SET);
  fread(this->pTexCoor, sizeof(sTexCoor), this->numTexCoor, pFile);


  for(int i = 0; i < this->numFrames; ++i)
    {
      frame = (sFrame*)(buffer + this->header->frameSize * i);
      pVertex = this->pVertices + this->numVertices  * i;
      pNormals = this->pLightNormals + this->numVertices * i;

      for(int j = 0; j < this->numVertices; ++j)
        {
          /* SPEEDUP: *(pVerts + i + 0) = (*(frame->pVertices + i + 0)...  */
           pVertex[j][0] = ((frame->pVertices[j].v[0] * frame->scale[0] ) + frame->translate[0] )* this->scaleFactor;
           pVertex[j][1] = ((frame->pVertices[j].v[2] * frame->scale[2]) + frame->translate[2]) * this->scaleFactor;
           pVertex[j][2] = (-1.0 * (frame->pVertices[j].v[1] * frame->scale[1] + frame->translate[1])) * this->scaleFactor;

          //printf("vertex %i/%i: (%f, %f, %f)\n", j, this->numVertices, pVertex[j][0], pVertex[j][1], pVertex[j][2]);

          pNormals[j] = frame->pVertices[j].lightNormalIndex;
        }
    }
    PRINTF(4)("Finished loading the md2 file\n");

  delete [] buffer;
  fclose(pFile);
}


/**
  \brief loads the skin/material stuff
* @param fileName: name of the skin file
  \return true if success
*/
bool MD2Data::loadSkin(const std::string& fileName)
{
  if( fileName.empty())
    {
      this->skinFileName = "";
      return false;
    }

  this->skinFileName = fileName;

  this->material.setName("md2ModelMaterial");
  this->material.setDiffuseMap(fileName);
  this->material.setIllum(3);
  this->material.setAmbient(1.0, 1.0, 1.0);
}
