/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Christian Meyer
   co-programmer: Benjamin Grauer

   2005-06-10: some naming conventions

//
//  splitLine
//  STL string tokenizer
//
//  Created by Clemens Wacha.
//  Version 1.0
//  Copyright (c) 2005 Clemens Wacha. All rights reserved.
//

*/


/**
 *  breaks a string into parts that were initially seperated by comma
 * @param string the string to break into substrings
*/

#include "substring.h"

#include <string.h>
#include <cassert>

SubString::SubString(const std::string& string, char splitter)
{
  char split[2];
  split[0] = splitter;
  split[1] = '\0';
  SubString::splitLine(this->strings, this->offsets,
                       string, split);
}

/**
 * Splits a String into a Substring removing all whiteSpaces
 * @param string the String to Split
 * @param whiteSpaces MUST BE __TRUE__
 *
 */
SubString::SubString(const std::string& string, bool whiteSpaces)
{
  SubString::splitLine(this->strings, this->offsets,
                      string);
}
SubString::SubString(const std::string& string, const std::string& splitters, char escapeChar,char safemode_char, char comment_char)
{
  SubString::splitLine(this->strings, this->offsets,
                       string, splitters, escapeChar, safemode_char);
}

/**
 * An empty String
 */
const std::string SubString::emptyString = "";



unsigned int SubString::split(const std::string& string, char splitter)
{
  this->offsets.clear();
  this->strings.clear();
  char split[2];
  split[0] = splitter;
  split[1] = '\0';
  SubString::splitLine(this->strings, this->offsets, string, split);
  return strings.size();
}


/**
 * Splits a String into a Substring removing all whiteSpaces
 * @param string the String to Split
 * @param whiteSpaces MUST BE __TRUE__
 *
 */
unsigned int SubString::split(const std::string& string, bool whiteSpaces)
{
  this->offsets.clear();
  this->strings.clear();
  SubString::splitLine(this->strings, this->offsets, string);
  return strings.size();
}

unsigned int SubString::split(const std::string& string, const std::string& splitters, char escapeChar,char safemode_char, char comment_char)
{
  this->offsets.clear();
  this->strings.clear();
  SubString::splitLine(this->strings, this->offsets,
                       string, splitters, escapeChar, safemode_char);
  return strings.size();
}


/**
 * @brief splits line into tokens and stores them in ret.
 * @param ret the Array, where the Splitted strings will be stored in
 * @param offsets an Array of Offsets, here the distance from the inputstring
 * to the beginning of the current token is stored
 * @param line the inputLine to split
 * @param delimiters a String of Delimiters (here the input will be splitted)
 * @param escape_char: Escape carater (escapes splitters)
 * @param safemode_char: the beginning of the safemode is marked with this
 * @param comment_char: the beginning of a comment is marked with this: (until the end of a Line)
 * @param start_state: the Initial state on how to parse the String.
 * @returns SPLIT_LINE_STATE the parser was in when returning
 *
 * Supports delimiters, escape characters,
 * ignores special  characters between safemode_char and between comment_char and linend '\n'.
 *
 */
SPLIT_LINE_STATE SubString::splitLine(std::vector<std::string>& ret, std::vector<unsigned int>& offsets,
                                      const std::string& line, const std::string& delimiters,
                                      char escape_char, char safemode_char, char comment_char,
                                      SPLIT_LINE_STATE start_state)
{
  SPLIT_LINE_STATE state = start_state;
  unsigned int i = 0;
  std::string token;

  if(start_state != SL_NORMAL && ret.size() > 0)
  {
    token = ret[ret.size()-1];
    ret.pop_back();
  }

  while(i < line.size())
  {
    switch(state)
    {
    case SL_NORMAL:
      if(line[i] == escape_char)
      {
        state = SL_ESCAPE;
      }
      else if(line[i] == safemode_char)
      {
        state = SL_SAFEMODE;
      }
      else if(line[i] == comment_char)
      {
        /// FINISH
        if(token.size() > 0)
        {
          ret.push_back(token);
          offsets.push_back(i);
          token.clear();
        }
        token += line[i];       // EAT
        state = SL_COMMENT;
      }
      else if(delimiters.find(line[i]) != std::string::npos)
      {
        // line[i] is a delimiter
        /// FINISH
        if(token.size() > 0)
        {
          ret.push_back(token);
          offsets.push_back(i);
          token.clear();
        }
      }
      else
      {
        token += line[i];       // EAT
      }
      break;
    case SL_ESCAPE:
      if(line[i] == 'n') token += '\n';
      else if(line[i] == 't') token += '\t';
      else if(line[i] == 'v') token += '\v';
      else if(line[i] == 'b') token += '\b';
      else if(line[i] == 'r') token += '\r';
      else if(line[i] == 'f') token += '\f';
      else if(line[i] == 'a') token += '\a';
      else if(line[i] == '?') token += '\?';
      else token += line[i];  // EAT
      state = SL_NORMAL;
      break;
    case SL_SAFEMODE:
      if(line[i] == safemode_char)
      {
        state = SL_NORMAL;
      }
      else if(line[i] == escape_char)
      {
        state = SL_SAFEESCAPE;
      }
      else
      {
        token += line[i];       // EAT
      }
      break;
    case SL_SAFEESCAPE:
      if(line[i] == 'n') token += '\n';
      else if(line[i] == 't') token += '\t';
      else if(line[i] == 'v') token += '\v';
      else if(line[i] == 'b') token += '\b';
      else if(line[i] == 'r') token += '\r';
      else if(line[i] == 'f') token += '\f';
      else if(line[i] == 'a') token += '\a';
      else if(line[i] == '?') token += '\?';
      else token += line[i];  // EAT
      state = SL_SAFEMODE;
      break;
    case SL_COMMENT:
      if(line[i] == '\n')
      {
        /// FINISH
        if(token.size() > 0)
        {
          ret.push_back(token);
          offsets.push_back(i);
          token.clear();
        }
        state = SL_NORMAL;
      }
      else
      {
        token += line[i];       // EAT
      }
      break;
    default:
      // nothing
      break;
    }
    i++;
  }

  /// FINISH
  if(token.size() > 0)
  {
    ret.push_back(token);
    offsets.push_back(i);
    token.clear();
  }
  return(state);
}

/**
 *  removes the object from memory
*/
SubString::~SubString()
{ }

/**
 * get a particular substring's offset
 * @param i the ID of the substring to get the offset from
 * @returns the offset or NULL if an invalid ID was given
 */
unsigned int SubString::getOffset(unsigned int i)
{
  if( i < this->offsets.size() && i >= 0)
    return this->offsets[i];
  else
    return 0;
}

/**
 * Some nice debug information about this SubString
 */
void SubString::debug() const
{
  printf("Substring-information::count=%d ::", this->strings.size());
  for (unsigned int i = 0; i < this->strings.size(); i++)
    printf("s%d='%s'::", i, this->strings[i].c_str());
  printf("\n");
}
