/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_LOADING

#include "util/loading/load_param.h"
#include "load_param_description.h"

#include <stdarg.h>

/**
 * Constructs a new LoadParameter
 * @param root the XML-element to load this Parameter from
 * @param paramName the Parameter to load
 * @param object the BaseObject, to load this parameter on to (will be cast to executor's Parameter)
 * @param executor the Executor, that executes the loading procedure.
 */
CLoadParam::CLoadParam(const TiXmlElement* root, const std::string& paramName, BaseObject* object, const Executor& executor, bool inLoadCycle)
  :  paramName(paramName), object(object)
{
  this->object = object;
  this->inLoadCycle = inLoadCycle;

  // determin the LoadString.
  if (likely(!inLoadCycle))
    this->loadElem = grabParameterElement(root, paramName);
  else if (paramName == root->Value())
    this->loadElem = (TiXmlElement*)root->FirstChild();
  else
    this->loadElem = NULL;

  // set the Executor.
  this->executor = executor.clone();

  //if (this->executor)
  //  this->executor->setName(paramName);
}

/**
 * This is a VERY SPECIAL deconsrtuctor.
 * It is made, so that it loads the Parameters on destruction.
 * meaning, if an Executor a valid Object exist, and all
 * Execution-Conditions are met, they are executed here.
 */
CLoadParam::~CLoadParam()
{
  if (likely(this->executor != NULL))
  {
    std::string loadString = "";
    if (this->loadElem != NULL &&  this->loadElem->ToText())
      loadString = this->loadElem->Value();
    if (likely(this->object != NULL) &&
        ( !loadString.empty() ||
          ((this->executor->getType() & Executor_NoLoadString) == Executor_NoLoadString)))
    {
      PRINTF(0)("Loading value '%s' with Parameters '%s' onto: %s::%s\n", this->paramName.c_str(), loadString.c_str(), this->object->getClassName(), this->object->getName());
      this->executor->execute(this->object, loadString);
    }
    delete this->executor;
  }
}

/**
 * @brief set the default values of the executor
 * @param value0 the first default value
 * @param value1 the second default value
 * @param value2 the third default value
 * @param value3 the fourth default value
 * @param value4 the fifth default value
 */
CLoadParam& CLoadParam::defaultValues(const MultiType& value0, const MultiType& value1,
                                      const MultiType& value2, const MultiType& value3,
                                      const MultiType& value4)
{
  assert(this->executor != NULL);
  this->executor->defaultValues(value0, value1, value2, value3, value4);

  return *this;
}



/**
 * @param descriptionText The text to set as a description for this Parameter
 * @returns a pointer to itself.
*/
CLoadParam& CLoadParam::describe(const std::string& descriptionText)
{
  if (LoadClassDescription::parametersDescription && this->paramDesc && this->paramDesc->getDescription().empty())
  {
    this->paramDesc->setDescription(descriptionText);
  }
  return *this;
}

// const LoadParamDescription* LoadParamDescription::getClass(const char* className)
// {
//   tIterator<LoadClassDescription>* iterator = LoadClassDescription::classList->getIterator();
//   LoadClassDescription* enumClassDesc = iterator->firstElement();
//   while (enumClassDesc)
//   {
//     if (!strcmp(enumClassDesc->className, classNameBegin, className))
//     {
//       delete iterator;
//       return enumClassDesc;
//     }
//     enumClassDesc = iterator->nextElement();
//   }
//   delete iterator;
//
//   return NULL;
// }




/*
 * @param object The object this Parameter is loaded too.
 * @param root: the XML-element to load this option from.
 * @param paramName: The name of the parameter loaded.
 * @param paramCount: how many parameters this loading-function takes
 * @param multi: if false LoadParam assumes only one occurence of this parameter in root, if true it assumes multiple occurences.
 * @param ...: the parameter information (1. Parameter, 2. Default Value for the Parameter, ...)
*/
/*LoadParam::LoadParam(const TiXmlElement* root, BaseObject* object, const char* paramName,
                             int paramCount, bool multi, const void* pointerToParam, ...)
{
  this->setClassID(CL_LOAD_PARAM, "LoadParam");
  this->executor = NULL;

  this->loadString = NULL;
  this->pointerToParam = pointerToParam;

  if (paramCount == 0 || this->pointerToParam != NULL)
    this->loadString = "none";
  else
{
      if (likely(!multi))
        this->loadString = grabParameter(root, paramName);
      else
{
          if (!strcmp(root->Value(), paramName))
{
              const TiXmlNode* val = root->FirstChild();
              if( val->ToText())
                this->loadString = val->Value();
}
}
}

  this->paramDesc = NULL;
  if (LoadClassDescription::parametersDescription)
{
    // locating the class
    this->classDesc = LoadClassDescription::addClass(object->getClassName());

    if ((this->paramDesc = this->classDesc->addParam(paramName)) != NULL)
{

      this->paramDesc->paramCount = paramCount;
      this->paramDesc->types = new int[paramCount];
      this->paramDesc->defaultValues = new char*[paramCount];

      va_list types;
      va_start (types, pointerToParam);
      char defaultVal[512];
      for(int i = 0; i < paramCount; i++)
{
        defaultVal[0] = '\0';
          // parameters parsed
        int tmpType = va_arg (types, int);
        this->paramDesc->types[i] = tmpType;
        switch (tmpType)
{
  case MT_INT:
            sprintf(defaultVal, "%d", va_arg(types, int));
            break;
//          case MT_LONG:
//            sprintf(defaultVal, "%0.3f", va_arg(types, l_LONG_TYPE));
//            break;
  case MT_FLOAT:
            sprintf(defaultVal, "%0.3f", va_arg(types, double));
            break;
  case MT_STRING:
            sprintf(defaultVal, "%s", va_arg(types, l_STRING_TYPE));
            break;
  case MT_EXT1:
            sprintf(defaultVal, "");
            break;
}
        this->paramDesc->defaultValues[i] = new char[strlen(defaultVal)+1];
        strcpy(this->paramDesc->defaultValues[i], defaultVal);
}
      va_end(types);

      int argCount = 0;
}
}
}*/










//////////////////////
// HELPER FUNCTIONS //
//////////////////////
/**
 * @param root: The XML-element to grab a parameter from
 * @param parameterName: the parameter to grab
 * @returns the Value of the parameter if found, NULL otherwise
*/
std::string grabParameter(const TiXmlElement* root, const std::string& parameterName)
{
  const TiXmlElement* element;
  const TiXmlNode* node;

  if (root == NULL)
    return "";

  element = root->FirstChildElement( parameterName);
  if( element == NULL) return "";

  node = element->FirstChild();
  while( node != NULL)
  {
    if( node->ToText()) return node->Value();
    node = node->NextSibling();
  }
  return "";
}

/**
 * @param root: The XML-element to grab a parameter from
 * @param parameterName: the parameter to grab
 * @returns the Element of the parameter if found, NULL otherwise
 */
const TiXmlElement* grabParameterElement(const TiXmlElement* root, const std::string& parameterName)
{
  const TiXmlElement* element;
  const TiXmlNode* node;

  if (root == NULL)
    return NULL;

  element = root->FirstChildElement( parameterName);
  if( element == NULL) return NULL;

  node = element->FirstChild();
  while( node != NULL)
  {
    if( node->ToText()) return (TiXmlElement*)node;
    node = node->NextSibling();
  }
  return NULL;
}



