/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Christian Meyer
   co-programmer: Benjamin Grauer
*/

/*!
 * @file factory.h
 * @brief A loadable object handler
*/


#ifndef _FACTORY_H
#define _FACTORY_H

class BaseObject;

#include "parser/tinyxml/tinyxml.h"
#include "base_object.h"
#include <vector>
#include <list>

/**
 * Creates a factory to a Loadable Class.
 * this should be used at the beginning of all the Classes that should be loadable (in the cc-file)
 */
#define CREATE_FACTORY(CLASS_NAME, CLASS_ID) \
    tFactory<CLASS_NAME>* global_##CLASS_NAME##_Factory = new tFactory<CLASS_NAME>(#CLASS_NAME, CLASS_ID)

// #define CREATE_DYNAMIC_FACTORY(CLASS_NAME, CLASS_ID) \
//     tFactory<CLASS_NAME>* global_##CLASS_NAME##_Dynamic_Factory = new DynamicLoader<CLASS_NAME>(#CLASS_NAME, CLASS_ID)

//! The Factory is a loadable object handler
class Factory : public BaseObject {

 public:
  virtual ~Factory ();

  static void deleteFactories();

  static  BaseObject* fabricate(const std::string& className);
  static  BaseObject* fabricate(ClassID classID);
  static  BaseObject* fabricate(const TiXmlElement* root = NULL);


  bool operator==(ClassID classID) const;
  bool operator==(const char* className) const;
  bool operator==(const std::string& className) const;

  protected:
    Factory (const std::string& factoryName, ClassID classID);
    virtual BaseObject* fabricateObject(const TiXmlElement* root = NULL) const = 0;

  protected:
    const ClassID                 classID;              //!< The Class-Identifyer of the Factory.
    const std::string             className;            //!< The name of the Class.
    static std::list<Factory*>*   factoryList;          //!< List of Registered Factories
};

/**
 *  @brief a factory that is able to load any kind of Object
 * (this is a Functor)
 */
template<class T> class tFactory : public Factory
{
 public:
 /**
  * @brief creates a new type Factory to enable the loading of T
  * @param factoryName the Name of the Factory to load.
  * @param classID the ID of the Class to be created.
  */
  tFactory (const char* factoryName, ClassID classID)
   : Factory(factoryName, classID)
  {  }

  private:
   /**
    * @brief fabricates an Object of type T, with the constructor T::T(const TiXmlElemnt*)
    * @param root the TiXmlElement T should load parameters from.
    * @return the newly fabricated T.
    */
    virtual BaseObject* fabricateObject(const TiXmlElement* root = NULL) const
    {
      return new T(root);
    }
};

#endif /* _FACTORY_H */

