/*!
 * @file executor.h
 * Definition of an Executor
 */

#ifndef _EXECUTOR_H
#define _EXECUTOR_H

#include "base_object.h"

#include "helper_functions.h"
#include "multi_type.h"
#include "substring.h"
#include "functor_list.h" //< MUST BE INCLUDED HERE AT LEAST ONCE.

//! an enumerator for the definition of the Type.
typedef enum {
  Executor_Objective         = 0x00000001,
  Executor_Static            = 0x00000002,

  Executor_NoLoadString      = 0x00000010,
} Executor_Type;

////////////////
// BASE CLASS //
////////////////
//! a BaseClass for all possible Executors
/**
 * An Executor is an Object, that is able to call Objects of Any type (class)
 * and execute a function with given parameters on it.
 *
 * The Executor is able to handle:
 *  Objects of any Class (Templated)
 *  Default Values
 *  Functions with up to 5 parameters (more seems useless)
 *  Functions with many types (see functor_list.h)
 */
class Executor: public BaseObject
{
  public:
    virtual ~Executor();

    virtual Executor* clone () const = 0;

    Executor* defaultValues(const MultiType& value0 = MT_NULL, const MultiType& value1 = MT_NULL,
                            const MultiType& value2 = MT_NULL, const MultiType& value3 = MT_NULL,
                            const MultiType& value4 = MT_NULL);

    /** executes a Command @param object the object to apply this to @param parameters the parameters the command takes */
    virtual void execute (BaseObject* object, const void* parameters = NULL) = 0;

    /** @returns the Type of this Function (either static or objective) */
    inline long getType() const { return this->functorType; };
    /** @returns the Count of Parameters this Executor takes */
    inline unsigned int getParamCount() const { return this->paramCount; };

    static void debug();

  protected:
    Executor(const MultiType& param0 = MT_NULL, const MultiType& param1 = MT_NULL,
             const MultiType& param2 = MT_NULL, const MultiType& param3 = MT_NULL,
             const MultiType& param4 = MT_NULL);

    void cloning(Executor* executor) const;

  protected:
    long                        functorType;      //!< The type of Function we've got (either static or objective).
    unsigned int                paramCount;       //!< the count of parameters.
    MultiType                   defaultValue[5];  //!< Default Values.
};

///////////////////////////////////////////////////
///////////////////////////////////////////////////

/////////////////////////////////
// MACRO DEFINITION EXTENSIONS //
/////////////////////////////////
//! where to chek for default BOOL values
#define   l_BOOL_DEFGRAB(i)         this->defaultValue[i].getBool()
//! where to chek for default INT values
#define   l_INT_DEFGRAB(i)          this->defaultValue[i].getInt()
//! where to chek for default UINT values
#define   l_UINT_DEFGRAB(i)         (unsigned int)this->defaultValue[i].getInt()
//! where to chek for default LONG values
#define   l_LONG_DEFGRAB(i)         (long)this->defaultValue[i].getInt()
//! where to chek for default FLOAT values
#define   l_FLOAT_DEFGRAB(i)        this->defaultValue[i].getFloat()
//! where to chek for default STRING values
#define   l_STRING_DEFGRAB(i)       this->defaultValue[i].getString()
//! where to chek for default CSTRING values
#define   l_CSTRING_DEFGRAB(i)      this->defaultValue[i].getCString()

//////////////////////////
// COMMAND REGISTRATION //
//////////////////////////
// EXECUTOR             can be redefined as Executor or ExecutorStatic
// EXECUTOREXECUTER     can be redefined too.
// EXECUTORINCLASS
// EXECUTORTYPE


///////////////////////
// FUNCTION POINTERS //
///////////////////////
#define ExecutorFunctionPoiter0() \
  void EXECUTORINCLASS(*functionPointer_0)();

#define ExecutorFunctionPoiter1(t1) \
  void EXECUTORINCLASS(*functionPointer_1_##t1)(t1##_TYPE);

#define ExecutorFunctionPoiter2(t1, t2) \
  void EXECUTORINCLASS(*functionPointer_2_##t1##_##t2)(t1##_TYPE, t2##_TYPE);


#define ExecutorFunctionPoiter3(t1, t2, t3) \
  void EXECUTORINCLASS(*functionPointer_3_##t1##_##t2##_##t3)(t1##_TYPE, t2##_TYPE, t3##_TYPE);

#define ExecutorFunctionPoiter4(t1, t2, t3, t4) \
  void EXECUTORINCLASS(*functionPointer_4_##t1##_##t2##_##t3##_##t4)(t1##_TYPE, t2##_TYPE, t3##_TYPE, t4##_TYPE);


#define ExecutorFunctionPoiter5(t1, t2, t3, t4, t5) \
  void EXECUTORINCLASS(*functionPointer_5_##t1##_##t2##_##t3##_##t4##_##t5)(t1##_TYPE, t2##_TYPE, t3##_TYPE, t4##_TYPE, t5##_TYPE); \


//////////////////
// CONSTRUCTORS //
/////////////////
//! creates a command that takes no parameters
#define ExecutorConstructor0() \
  EXECUTOR(void EXECUTORINCLASS(*function)()) \
  : Executor(0) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_0 = function; \
  }

//! creates a command that takes one parameter
#define ExecutorConstructor1(t1) \
  EXECUTOR(void EXECUTORINCLASS(*function)(t1##_TYPE)) \
  : Executor(t1##_PARAM) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_1_##t1 = function; \
  }

//! creates a command that takes two parameters
#define ExecutorConstructor2(t1,t2) \
  EXECUTOR(void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE)) \
  : Executor(t1##_PARAM, t2##_PARAM) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_2_##t1##_##t2 = function; \
  }

//! creates a command that takes three parameter
#define ExecutorConstructor3(t1,t2,t3) \
  EXECUTOR(void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE, t3##_TYPE)) \
  : Executor(t1##_PARAM, t2##_PARAM, t3##_PARAM) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_3_##t1##_##t2##_##t3 = function; \
  }

//! creates a command that takes four parameter
#define ExecutorConstructor4(t1,t2,t3,t4) \
  EXECUTOR(void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE, t3##_TYPE, t4##_TYPE)) \
  : Executor(t1##_PARAM, t2##_PARAM, t3##_PARAM, t4##_PARAM) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_4_##t1##_##t2##_##t3##_##t4 = function; \
  }

//! creates a command that takes five parameter
#define ExecutorConstructor5(t1,t2,t3,t4,t5) \
  EXECUTOR(void EXECUTORINCLASS(*function)(t1##_TYPE, t2##_TYPE, t3##_TYPE, t4##_TYPE, t5##_TYPE)) \
  : Executor(t1##_PARAM, t2##_PARAM, t3##_PARAM, t4##_PARAM, t5##_PARAM) \
  { \
    this->functorType = EXECUTORTYPE; \
    this->fp.functionPointer_5_##t1##_##t2##_##t3##_##t4##_##t5 = function; \
  }

///////////////
// EXECUTION //
///////////////
//! execute-macro for functions with no parameters
#define ExecutorExecute0() \
  if (this->paramCount == 0) \
    EXECUTOREXECUTER(_0)()

//! execute-macro for functions with one parameter
#define ExecutorExecute1(t1) \
   else if (this->paramCount == 1 && this->defaultValue[0] == t1##_PARAM) \
    EXECUTOREXECUTER(_1_##t1)(t1##_FUNC((const char*)parameters, t1##_DEFGRAB(0)))

//! execute-macro for functions with two parameters
#define ExecutorExecute2(t1,t2) \
   else if (this->paramCount == 2 && this->defaultValue[0] == t1##_PARAM && this->defaultValue[1] == t2##_PARAM) \
    EXECUTOREXECUTER(_2_##t1##_##t2)(t1##_FUNC(sub.getString(0), t1##_DEFGRAB(0)), t2##_FUNC(sub.getString(1), t2##_DEFGRAB(1)))

//! execute-macro for functions with three parameters
#define ExecutorExecute3(t1,t2,t3) \
   else if (this->paramCount == 3 && this->defaultValue[0] == t1##_PARAM && this->defaultValue[1] == t2##_PARAM && this->defaultValue[2] == t3##_PARAM) \
    EXECUTOREXECUTER(_3_##t1##_##t2##_##t3)(t1##_FUNC(sub.getString(0), t1##_DEFGRAB(0)), t2##_FUNC(sub.getString(1), t2##_DEFGRAB(1)), t3##_FUNC(sub.getString(2), t3##_DEFGRAB(2)))

//! execute-macro for functions with four parameters
#define ExecutorExecute4(t1,t2,t3,t4) \
   else if (this->paramCount == 4 && this->defaultValue[0] == t1##_PARAM && this->defaultValue[1] == t2##_PARAM && this->defaultValue[2] == t3##_PARAM && this->defaultValue[3] == t4##_PARAM) \
    EXECUTOREXECUTER(_4_##t1##_##t2##_##t3##_##t4)(t1##_FUNC(sub.getString(0), t1##_DEFGRAB(0)), t2##_FUNC(sub.getString(1), t2##_DEFGRAB(1)), t3##_FUNC(sub.getString(2), t3##_DEFGRAB(2)), t4##_FUNC(sub.getString(3), t4##_DEFGRAB(3))) \


//! execute-macro for functions with five parameters
#define ExecutorExecute5(t1,t2,t3,t4,t5) \
   else if (this->paramCount == 5 && this->defaultValue[0] == t1##_PARAM && this->defaultValue[1] == t2##_PARAM && this->defaultValue[2] == t3##_PARAM && this->defaultValue[3] == t4##_PARAM && this->defaultValue[4] == t5##_PARAM) \
    EXECUTOREXECUTER(_5_##t1##_##t2##_##t3##_##t4##_##t5)(t1##_FUNC(sub.getString(0), t1##_DEFGRAB(0)), t2##_FUNC(sub.getString(1), t2##_DEFGRAB(1)), t3##_FUNC(sub.getString(2), t3##_DEFGRAB(2)), t4##_FUNC(sub.getString(3), t4##_DEFGRAB(3)), t5##_FUNC(sub.getString(4), t5##_DEFGRAB(4)))





//////////\//////////
// DYNAMIC FUNCTOR //
///////////\/////////
#ifdef FUNCTOR_LIST
#undef FUNCTOR_LIST
#endif
#ifdef EXECUTOR
#undef EXECUTOR
#endif
#define EXECUTOR                       ExecutorObjective
#ifdef EXECUTOREXECUTER
#undef EXECUTOREXECUTER
#endif
#define EXECUTOREXECUTER(nameExt)      (dynamic_cast<T*>(object)->*(fp.functionPointer##nameExt))
#ifdef EXECUTORINCLASS
#undef EXECUTORINCLASS
#endif
#define EXECUTORINCLASS(FUNCTION)      (T::FUNCTION)
#ifdef EXECUTORTYPE
#undef EXECUTORTYPE
#endif
#define EXECUTORTYPE                   Executor_Objective
//! keeps information about a Executor
template<class T> class ExecutorObjective : public Executor
{
  public:
    ExecutorObjective() : Executor() { };
    // COPY constuctor (virtual version)
    virtual Executor* clone () const
    {
      ExecutorObjective<T>* executor = new ExecutorObjective<T>();
      this->cloning(executor);
      executor->fp = this->fp;
      return executor;
    }

//! FUNCTOR_LIST is the List of CommandConstructors
#define FUNCTOR_LIST(x) ExecutorConstructor ## x
#include "functor_list.h"
#undef FUNCTOR_LIST

  private:
//! FUNCTOR_LIST is the List of FunctionPointers
    union FunctionPointers {
#define FUNCTOR_LIST(x) ExecutorFunctionPoiter ## x
#include "functor_list.h"
#undef FUNCTOR_LIST
    } fp;

    virtual void execute (BaseObject* object, const void* parameters = NULL)
    {
      SubString sub(((const std::string*) parameters)->c_str(), " \n\t,", '\\');
//! FUNCTOR_LIST is the List of Executive Functions
#define FUNCTOR_LIST(x) ExecutorExecute ## x
#include "functor_list.h"
#undef FUNCTOR_LIST
    }
};


////////////////////
// STATIC FUNCTOR //
////////////////////
#ifdef FUNCTOR_LIST
#undef FUNCTOR_LIST
#endif
#ifdef EXECUTOR
#undef EXECUTOR
#endif
#define EXECUTOR                      ExecutorStatic
#ifdef EXECUTOREXECUTER
#undef EXECUTOREXECUTER
#endif
#define EXECUTOREXECUTER(nameExt)     fp.functionPointer##nameExt
#ifdef EXECUTORINCLASS
#undef EXECUTORINCLASS
#endif
#define EXECUTORINCLASS(FUNCTION)     (FUNCTION)
#ifdef EXECUTORTYPE
#undef EXECUTORTYPE
#endif
#define EXECUTORTYPE                   Executor_Static

//! keeps information about a Executor, that points to a Static Function
template<class T> class ExecutorStatic : public Executor
{
  public:
    ExecutorStatic() : Executor() { };
    // COPY constuctor
    virtual Executor* clone () const
    {
      ExecutorStatic<T>* executor = new ExecutorStatic<T>();
      this->cloning(executor);
      executor->fp = this->fp;
      return executor;
    }

//! FUNCTOR_LIST is the List of CommandConstructors
#define FUNCTOR_LIST(x) ExecutorConstructor ## x
#include "functor_list.h"
#undef FUNCTOR_LIST

  private:
//! FUNCTOR_LIST is the List of FunctionPointers
    union FunctionPointers {
#define FUNCTOR_LIST(x) ExecutorFunctionPoiter ## x
#include "functor_list.h"
#undef FUNCTOR_LIST
    } fp;


    virtual void execute (BaseObject* object, const void* parameters = NULL)
    {
  SubString sub((const char*)parameters, " \n\t,");
//! FUNCTOR_LIST is the List of Executive Functions
#define FUNCTOR_LIST(x) ExecutorExecute ## x
#include "functor_list.h"
#undef FUNCTOR_LIST
    }
};

#endif /* _EXECUTOR_H */
