/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...


   -------------------------------------------------------------------
   The source of this file comes stright from http://www.devmaster.net
   Thanks a lot for the nice work, and the easy portability to our Project.
*/

#include <iostream>

#include "ogg_player.h"

#include "sound_engine.h"

#include "debug.h"

/**
 * initializes an Ogg-player from a file
 * @param fileName the file to load
 */
OggPlayer::OggPlayer(const std::string& fileName)
{
  this->setClassID(CL_SOUND_OGG_PLAYER, "OggPlayer");

  this->trackLoaded = false;
  this->source = 0;
  this->buffers[0] = 0;
  this->buffers[1] = 0;

  if (!fileName.empty())
  {
    if (this->open(fileName))
      this->setName(fileName);
  }
}

OggPlayer::~OggPlayer()
{
  //this->release();
}

/**
 * opens a file for playback
 * @param fileName the file to open
 */
bool OggPlayer::open(const std::string& fileName)
{
  if (this->buffers[0] == 0)
    alGenBuffers(2, this->buffers);
  SoundEngine::checkError("Allocating Buffers", __LINE__);

  if (this->source == 0)
    SoundEngine::getInstance()->popALSource(this->source);
  if (this->source == 0)
  {
    this->trackLoaded = false;
    return false;
  }

  int result;

  if(!(oggFile = fopen(fileName.c_str(), "rb")))
  {
    PRINTF(2)("Could not open Ogg file.");
    return false;
  }

  if((result = ov_open(oggFile, &oggStream, NULL, 0)) < 0)
  {
    PRINTF(2)("Could not open Ogg stream. %s", errorString(result));
    fclose(oggFile);
    return false;
  }

  vorbisInfo = ov_info(&oggStream, -1);
  vorbisComment = ov_comment(&oggStream, -1);

  if(vorbisInfo->channels == 1)
    format = AL_FORMAT_MONO16;
  else
    format = AL_FORMAT_STEREO16;

  alSource3f(source, AL_POSITION,        0.0, 0.0, 0.0);
  alSource3f(source, AL_VELOCITY,        0.0, 0.0, 0.0);
  alSource3f(source, AL_DIRECTION,       0.0, 0.0, 0.0);
  alSourcef (source, AL_ROLLOFF_FACTOR,  0.0          );
  alSourcei (source, AL_SOURCE_RELATIVE, AL_TRUE      );
  alSourcef (source, AL_GAIN,            SoundEngine::getInstance()->getMusicVolume());

  this->trackLoaded = true;
  return true;
}

/**
 * releases a stream
 */
void OggPlayer::release()
{
  if (!this->trackLoaded)
    return;
  alSourceStop(source);
  empty();
  SoundEngine::getInstance()->pushALSource(source);
  this->source = 0;
  check();
  alDeleteBuffers(2, buffers);
  this->buffers[0] = 0;
  this->buffers[1] = 0;
  check();

  ov_clear(&oggStream);
  this->trackLoaded = false;
}


/**
 * plays back the sound
 * @return true if running, false otherwise
 */
bool OggPlayer::playback()
{
  if (!this->trackLoaded)
    return false;

  if(playing())
    return true;

  if(!stream(buffers[0]))
    return false;

  if(!stream(buffers[1]))
    return false;

  alSourceQueueBuffers(source, 2, buffers);
  alSourcePlay(source);

  return true;
}

/**
 *
 * @returns true if the file is playing
 */
bool OggPlayer::playing()
{
  if (!this->trackLoaded)
    return false;
  ALenum state;

  alGetSourcei(this->source, AL_SOURCE_STATE, &state);

  return (state == AL_PLAYING);
}

/**
 * updates the stream, this has to be done every few parts of a second, for sound-consistency
 * @returns true, if the Sound is playing flawlessly
 */
bool OggPlayer::update()
{
  if (unlikely(!this->trackLoaded))
    return false;

  int processed;
  bool active = true;

  alGetSourcei(source, AL_BUFFERS_PROCESSED, &processed);

  while(processed--)
  {
    ALuint buffer;

    alSourceUnqueueBuffers(source, 1, &buffer);
    check();

    active = stream(buffer);

    alSourceQueueBuffers(source, 1, &buffer);
    check();
  }

  return active;
}

/**
 * gets a new Stream from buffer
 * @param buffer the buffer to get the stream from
 * @return true, if everything worked as planed
 */
bool OggPlayer::stream(ALuint buffer)
{
  if (!this->trackLoaded)
    return false;
  char pcm[BUFFER_SIZE];
  int  size = 0;
  int  section;
  int  result;

  while(size < BUFFER_SIZE)
  {
    result = ov_read(&oggStream, pcm + size, BUFFER_SIZE - size, 0, 2, 1, &section);

    if(result > 0)
      size += result;
    else
      if(result < 0)
        throw errorString(result);
      else
        break;
  }

  if(size == 0)
    return false;

  alBufferData(buffer, format, pcm, size, vorbisInfo->rate);
  check();

  return true;
}


/**
 * empties the buffers
 */
void OggPlayer::empty()
{
  int queued;

  alGetSourcei(source, AL_BUFFERS_QUEUED, &queued);

  while(queued--)
  {
    ALuint buffer;

    alSourceUnqueueBuffers(source, 1, &buffer);
    check();
  }
}


/**
 * displays some info about the ogg-file
 */
void OggPlayer::debug()
{
  cout
  << "version         " << vorbisInfo->version         << "\n"
  << "channels        " << vorbisInfo->channels        << "\n"
  << "rate (hz)       " << vorbisInfo->rate            << "\n"
  << "bitrate upper   " << vorbisInfo->bitrate_upper   << "\n"
  << "bitrate nominal " << vorbisInfo->bitrate_nominal << "\n"
  << "bitrate lower   " << vorbisInfo->bitrate_lower   << "\n"
  << "bitrate window  " << vorbisInfo->bitrate_window  << "\n"
  << "\n"
  << "vendor " << vorbisComment->vendor << "\n";

  for(int i = 0; i < vorbisComment->comments; i++)
    cout << "   " << vorbisComment->user_comments[i] << "\n";

  cout << endl;
}

/**
 * checks for errors
 */
void OggPlayer::check()
{
  int error = alGetError();
  if(error != AL_NO_ERROR)
    PRINTF(2)("OpenAL error was raised.");
}

/**
 * returns errors
 * @param code the error-code
 * @return the error as a String
 */
const char* OggPlayer::errorString(int code)
{
  switch(code)
  {
    case OV_EREAD:
      return ("Read from media.");
    case OV_ENOTVORBIS:
      return ("Not Vorbis data.");
    case OV_EVERSION:
      return ("Vorbis version mismatch.");
    case OV_EBADHEADER:
      return ("Invalid Vorbis header.");
    case OV_EFAULT:
      return ("Internal logic fault (bug or heap/stack corruption.");
    default:
      return ("Unknown Ogg error.");
  }
}
