/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_SOUND

#include "sound_buffer.h"

#include "sound_engine.h"

#include "sdlincl.h"
#include <cassert>

using namespace std;

//////////////////
/* SOUND-BUFFER */
//////////////////
/**
 *  Creates a Soundbuffer out of an inputfile
 * @param fileName The name of the File
 */
SoundBuffer::SoundBuffer(const std::string& fileName)
{
  this->setClassID(CL_SOUND_BUFFER, "SoundBuffer");
  this->setName(fileName);

  // generate a Buffer
  alGenBuffers(1, &this->bufferID);
  SoundEngine::checkError("Generate Buffer", __LINE__);
  this->loadWAV(fileName);
}

SoundBuffer::~SoundBuffer()
{
  //  SoundEngine::getInstance()->removeBuffer(this);
  alDeleteBuffers(1, &this->bufferID);
}

/**
 * @brief loads a Waveform from the local fileSystem into this Source.
 * @param fileName the Name of the File to Load.
 * @returns true on success.
 */
bool SoundBuffer::loadWAV(const std::string& fileName)
{
  SDL_AudioSpec wavSpec;
  Uint32 wavLength;
  Uint8 *wavBuffer;

  /* Load the WAV */
  if( SDL_LoadWAV(fileName.c_str(), &wavSpec, &wavBuffer, &wavLength) == NULL)
  {
    PRINTF(2)("Could not open %s: %s\n", fileName.c_str(), SDL_GetError());
    return false;
  }

  alBufferData(this->bufferID, SoundBuffer::sdlAudioSpecToAlFormat(&wavSpec), wavBuffer, wavLength, wavSpec.freq);
  SDL_FreeWAV(wavBuffer);
  if (SoundEngine::checkError("Could not load Wave file", __LINE__))
    return true;
  else
    return false;
}

/**
 * @brief converts an SDL_AudioSpec into a valid OpenAL AUDIO_FORMAT enumerator
 * @param audiospec the AudioSpec to convert.
 * @returns the AL_FORMAT
 */
ALenum SoundBuffer::sdlAudioSpecToAlFormat(const SDL_AudioSpec* audiospec)
{
  assert (audiospec != NULL);
  bool stereo = true;
  bool is16Bit = true;
  if (audiospec->format == AUDIO_U8 || audiospec->format == AUDIO_S8)
    is16Bit = false;
  if (audiospec->channels == 1)
    stereo = false;

  if (!stereo && !is16Bit)
    return AL_FORMAT_MONO8;
  else if (!stereo && is16Bit)
    return AL_FORMAT_MONO16;
  else if (stereo && !is16Bit)
    return AL_FORMAT_STEREO8;
  else if (stereo && is16Bit)
    return AL_FORMAT_STEREO16;
}

