/*!
 * @file particle_engine.h
  *  Definition of the ParticleEngine
*/

#ifndef _PARTICLE_ENGINE_H
#define _PARTICLE_ENGINE_H

#include "base_object.h"

#include "particle_system.h"
#include "particle_emitter.h"
#include <list>

#include "parser/tinyxml/tinyxml.h"

// FORWARD DECLARATION

//! A ParticleConnection enables us to emitt from any emitter into any other particleSystem
typedef struct ParticleConnection
{
  ParticleEmitter*    emitter;     //!< The emitter to emit system from.
  ParticleSystem*     system;      //!< The Particles emitted from emitter.
};

//! The physicsEngine handles and stores Systems and Emitters.
/**
   It is responsible for driving on the Particles (tick)
   It draw particles (draw)
   and it emitts particles into the system
*/
class ParticleEngine : public BaseObject {

 public:
  virtual ~ParticleEngine();
  /** @returns a Pointer to the only object of this Class */
  inline static ParticleEngine* getInstance() { if (!singletonRef) singletonRef = new ParticleEngine();  return singletonRef; };

  virtual void loadParams(const TiXmlElement* root);

  void tick(float dt);
  void draw() const;

  void addSystem(ParticleSystem* system);
  void addEmitter(ParticleEmitter* emitter);
  void addConnection(ParticleEmitter* emitter, ParticleSystem* system);
  void addConnection(const char* emitter, const char* system);


  bool removeSystem(ParticleSystem* system);
  bool removeEmitter(ParticleEmitter* emitter);
  bool breakConnection(ParticleConnection* connection);
  bool breakConnection(ParticleEmitter* emitter, ParticleSystem* system);
  unsigned int breakConnections(ParticleEmitter* emitter);
  unsigned int breakConnections(ParticleSystem* system);

   ParticleSystem* getSystemByNumber(unsigned int number) const;
   ParticleEmitter* getEmitterByNumber(unsigned int number) const;

  void debug();

 private:
  ParticleEngine();
  static ParticleEngine* singletonRef;        //!< The reference to the engine.

  std::list<ParticleSystem> systemList;          //!< A list of Systems handled by the ParticleEngine.
  std::list<ParticleEmitter> emitterList;        //!< A list of Emitters handled by the ParticleEngine.

  tList<ParticleConnection>* connectionList;  //!< A list of Connections between Systems and Emitters.
};

#endif /* _PARTICLE_ENGINE_H */
