/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer:

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "executor/executor.h"
#include "md2_creature.h"

#include "objModel.h"
#include "md2Model.h"
#include "resource_manager.h"
#include "state.h"

#include "weapons/weapon_manager.h"
#include "weapons/test_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"

#include "factory.h"
#include "key_mapper.h"
#include "event_handler.h"

#include "graphics_engine.h"

using namespace std;

CREATE_FACTORY(MD2Creature, CL_MD2_CREATURE);

/**
 *  destructs the MD2Creature, deletes alocated memory
 */
MD2Creature::~MD2Creature ()
{
  this->setPlayer(NULL);
}

/**
 * loads a MD2Creatures information from a specified file.
 * @param fileName the name of the File to load the MD2Creature from (absolute path)
 */
MD2Creature::MD2Creature(const char* fileName)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for md2 creature.\n", fileName);
    return;
  }

  this->loadParams(doc.RootElement());
  this->toList(OM_GROUP_01);
}

/**
 *  creates a new MD2Creature from Xml Data
 * @param root the xml element containing MD2Creature data

   @todo add more parameters to load
*/
MD2Creature::MD2Creature(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);

  //weapons:
  Weapon* wpRight = new TestGun(0);
  wpRight->setName("testGun Right");
  Weapon* wpLeft = new TestGun(1);
  wpLeft->setName("testGun Left");
  Weapon* cannon = dynamic_cast<Weapon*>(Factory::fabricate(CL_CANNON));

  cannon->setName("BFG");

   this->getWeaponManager()->addWeapon(wpLeft, 1, 0);
   this->getWeaponManager()->addWeapon(wpRight,1 ,1);
//   this->getWeaponManager()->addWeapon(cannon, 0, 6);

  //this->getWeaponManager()->addWeapon(turret, 3, 0);

  this->getWeaponManager()->changeWeaponConfig(0);
}


/**
 * initializes a MD2Creature
 */
void MD2Creature::init()
{
//  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->setClassID(CL_MD2_CREATURE, "MD2Creature");

  PRINTF(4)("MD2CREATURE INIT\n");

  this->mouseDirX *= Quaternion( M_PI * 0.75f, Vector(0,1,0));

  bUp = bDown = bLeft = bRight = bAscend = bDescend = bRollL = bRollR = bStrafeL = bStrafeR = bJump = false;
  bFire = false;
  xMouse = yMouse = 0;
  mouseSensitivity = 0.003;
  airViscosity = 0.0;
  cycle = 0.0;

  this->cameraConnNode.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->cameraConnNode.setName("CameraConnectorNode");
  this->addChild(&this->cameraConnNode);
  this->cameraConnNode.addChild(State::getCameraTargetNode());
  this->cameraConnNode.addChild(State::getCameraNode());
  State::getCameraTargetNode()->setRelCoor(10,0,0);

  travelSpeed = 15.0;
  this->velocity = Vector(0.0,0.0,0.0);

//   GLGuiButton* button = new GLGuiPushButton();
//   button->show();
//   button->setLabel("orxonox");
//   button->setBindNode(this);

  //add events to the eventlist
  this->registerEvent(SDLK_w);
  this->registerEvent(SDLK_s);
  this->registerEvent(SDLK_a);
  this->registerEvent(SDLK_d);
  this->registerEvent(SDLK_SPACE);
  this->registerEvent(SDLK_q);
  this->registerEvent(SDLK_e);
  this->registerEvent(KeyMapper::PEV_FIRE1);
  this->registerEvent(KeyMapper::PEV_NEXT_WEAPON);
  this->registerEvent(KeyMapper::PEV_PREVIOUS_WEAPON);
  this->registerEvent(EV_MOUSE_MOTION);



  this->getWeaponManager()->setSlotCount(7);

  this->getWeaponManager()->setSlotPosition(0, Vector(-0.5, .2, -1.9));
  this->getWeaponManager()->setSlotCapability(0, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager()->setSlotPosition(1, Vector(-0.5, .2, 1.9));
  this->getWeaponManager()->setSlotCapability(1, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager()->setSlotPosition(2, Vector(-1.5, .5, -.5));
  this->getWeaponManager()->setSlotDirection(2, Quaternion(-M_PI_4*.5, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(3, Vector(-1.5, .5, .5));
  this->getWeaponManager()->setSlotDirection(3, Quaternion(M_PI_4*.5, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(4, Vector(-1.5, -.5, .5));
  this->getWeaponManager()->setSlotDirection(4, Quaternion(-M_PI_4*.5+M_PI, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(5, Vector(-1.5, -.5, -.5));
  this->getWeaponManager()->setSlotDirection(5, Quaternion(+M_PI_4*.5-M_PI, Vector(1,0,0)));
//
   this->getWeaponManager()->setSlotPosition(6, Vector(-1, 0.0, 0));
   this->getWeaponManager()->setSlotCapability(6, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
   //

   this->cameraConnNode.addChild(this->getWeaponManager()->getFixedTarget());
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);

  this->getWeaponManager()->getFixedTarget()->setRelCoor(10,0,0);

}


void MD2Creature::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( true);
  this->attachCamera();

}

void MD2Creature::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);


}

/**
 * loads the Settings of a MD2Creature from an XML-element.
 * @param root the XML-element to load the MD2Creature's properties from
 */
void MD2Creature::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);
}


/**
 *  effect that occurs after the MD2Creature is spawned
*/
void MD2Creature::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}

/**
 *  the action occuring if the MD2Creature left the game
*/
void MD2Creature::leftWorld ()
{}

/**
 *  this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void MD2Creature::collidesWith(WorldEntity* entity, const Vector& location)
{
  PRINTF(0)("Collided with the md2 model\n");
}

/**
 *  draws the MD2Creature after transforming it.
*/
void MD2Creature::draw () const
{
  if (this->getCurrentPlayer() != NULL)
    WorldEntity::draw();
//  this->cameraConnNode.debugDraw(0);
}


/**
 *  the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void MD2Creature::tick (float time)
{
  Playable::tick(time);
  if( likely(this->getModel(0) != NULL))
    ((MD2Model*)this->getModel(0))->tick(time);


  // MD2Creature controlled movement
  this->calculateVelocity(time);
  Vector move = this->velocity * time;
  this->shiftCoor (move);

  if( this->bJump)
  {
    ((MD2Model*)this->getModel(0))->setAnim(JUMP);
  }
  else if( this->bFire)
  {
    if( ((MD2Model*)this->getModel(0))->getAnim() != ATTACK) ((MD2Model*)this->getModel(0))->setAnim(ATTACK);
  }
  else if( fabs(move.len()) > 0.0f)
  {
    if( ((MD2Model*)this->getModel(0))->getAnim() != RUN) ((MD2Model*)this->getModel(0))->setAnim(RUN);
  }
  else
  {
    if( ((MD2Model*)this->getModel(0))->getAnim() != STAND) ((MD2Model*)this->getModel(0))->setAnim(STAND);
  }


  //orient the MD2Creature in direction of the mouse
  this->setAbsDir(mouseDirX);
  this->cameraConnNode.setRelDir(mouseDirY);
}


/**
 *  calculate the velocity
 * @param time the timeslice since the last frame
*/
void MD2Creature::calculateVelocity (float time)
{
  Vector accel(0.0, 0.0, 0.0);
  /*
  Vector rot(0.0, 0.0, 0.0); // wird bentigt fr Helicopter
  */
  //float rotVal = 0.0;
  /* FIXME: calculating the direction and orthDirection every timeSlice is redundant! save it somewhere */
  /* calculate the direction in which the craft is heading  */

  if( this->bUp )
   {
     accel += (this->getAbsDirX())*2;
   }

  if( this->bDown )
   {
     accel -= (this->getAbsDirX())*2;
   }

  if( this->bLeft/* > -this->getRelCoor().z*2*/)
  {
    this->shiftDir(Quaternion(time, Vector(0,1,0)));
  }
  if( this->bRight /* > this->getRelCoor().z*2*/)
  {
    this->shiftDir(Quaternion(-time, Vector(0,1,0)));
  }


  if( this->bStrafeL /* > -this->getRelCoor().z*2*/)
  {
    accel -= this->getAbsDirZ() * 2.0f;
  }
  if( this->bStrafeR /* > this->getRelCoor().z*2*/)
  {
    accel += this->getAbsDirZ() * 2.0f;
  }
  if (this->bAscend )
  {
    this->shiftDir(Quaternion(time, Vector(0,0,1)));
  }
  if (this->bDescend )
  {
    this->shiftDir(Quaternion(-time, Vector(0,0,1)));
  }

  velocity = accel * 20.0f;
  //rot.normalize();
  //this->setRelDirSoft(Quaternion(rotVal, rot), 5);
}



/**
 * @todo switch statement ??
 */
void MD2Creature::process(const Event &event)
{
  Playable::process(event);
  if( event.type == SDLK_a)
      this->bStrafeL = event.bPressed;
  else if( event.type == SDLK_d)
      this->bStrafeR = event.bPressed;
  else if( event.type == SDLK_w)
    this->bUp = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == SDLK_s)
    this->bDown = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == SDLK_SPACE)
    this->bJump = event.bPressed;
  else if( event.type == EV_MOUSE_MOTION)
  {
    this->xMouse = event.xRel;
    this->yMouse = event.yRel;
    this->mouseDirX *= Quaternion(-M_PI/4*this->xMouse*mouseSensitivity, Vector(0,1,0));
    this->mouseDirY *= Quaternion(-M_PI/4*this->yMouse*mouseSensitivity, Vector(0,0,1));
//     if( xMouse*xMouse < 0.9)
//      this->setAbsDir(mouseDir);
  }
}
