/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Patrick Boenzli
   co-programmer: Christian Meyer
   co-programmer: Benjamin Grauer
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD

#include "game_world.h"
#include "game_world_data.h"

#include "resource_manager.h"
#include "state.h"
#include "class_list.h"
#include "substring.h"

#include "game_loader.h"

#include "p_node.h"
#include "world_entity.h"
#include "player.h"
#include "camera.h"
#include "environment.h"
#include "terrain.h"
#include "test_entity.h"
#include "terrain.h"
#include "md2Model.h"
#include "world_entities/projectiles/projectile.h"
#include "npcs/npc_test1.h"
#include "playable.h"

#include "light.h"

#include "factory.h"
#include "fast_factory.h"
#include "load_param.h"
#include "shell_command.h"

#include "graphics_engine.h"
#include "event_handler.h"
#include "sound_engine.h"
#include "cd_engine.h"
#include "network_manager.h"
#include "physics_engine.h"
#include "fields.h"

#include "glmenu_imagescreen.h"
#include "shell.h"

#include "animation_player.h"
#include "animation3d.h"

#include "ogg_player.h"
#include "shader.h"

#include "game_rules.h"

using namespace std;


SHELL_COMMAND(speed, GameWorld, setSpeed);
SHELL_COMMAND_STATIC(togglePNodeVisibility, GameWorld, GameWorld::togglePNodeVisibility);
SHELL_COMMAND_STATIC(toggleBVVisibility, GameWorld, GameWorld::toggleBVVisibility);



GameWorld::GameWorld()
    : StoryEntity()
{
  this->setClassID(CL_GAME_WORLD, "GameWorld");
  this->setName("Preloaded World - no name yet");

  this->gameTime = 0.0f;
  this->setSpeed(1.0f);
  this->shell = NULL;

  this->showPNodes = false;
  this->showBV = false;

  this->dataXML = NULL;
}

/**
 *  remove the GameWorld from memory
 *
 *  delete everything explicitly, that isn't contained in the parenting tree!
 *  things contained in the tree are deleted automaticaly
 */
GameWorld::~GameWorld ()
{
  PRINTF(4)("Deleted GameWorld\n");
}



/**
 * loads the parameters of a GameWorld from an XML-element
 * @param root the XML-element to load from
 */
void GameWorld::loadParams(const TiXmlElement* root)
{
  StoryEntity::loadParams(root);

  PRINTF(4)("Loaded GameWorld specific stuff\n");
}


/**
 * this is executed just before load
 *
 * since the load function sometimes needs data, that has been initialized
 * before the load and after the proceeding storyentity has finished
*/
ErrorMessage GameWorld::init()
{
  this->cycle = 0;
  /* init the world interface */
  this->shell = new Shell();

  State::setCurrentStoryEntity(dynamic_cast<StoryEntity*>(this));
  this->dataTank->init();
}


/**
 *  loads the GameWorld by initializing all resources, and set their default values.
 */
ErrorMessage GameWorld::loadData()
{
  this->displayLoadScreen();

  PRINTF(0)("Loading the GameWorld\n");

  PRINTF(3)("> Loading world: '%s'\n", getLoadFile());
  TiXmlElement* element;
  GameLoader* loader = GameLoader::getInstance();

  if( getLoadFile() == NULL)
  {
    PRINTF(1)("GameWorld has no path specified for loading\n");
    return (ErrorMessage){213,"Path not specified","GameWorld::load()"};
  }

  TiXmlDocument* XMLDoc = new TiXmlDocument( getLoadFile());
  // load the xml world file for further loading
  if( !XMLDoc->LoadFile())
  {
    PRINTF(1)("loading XML File: %s @ %s:l%d:c%d\n", XMLDoc->ErrorDesc(), this->getLoadFile(), XMLDoc->ErrorRow(), XMLDoc->ErrorCol());
    delete XMLDoc;
    return (ErrorMessage){213,"XML File parsing error","GameWorld::load()"};
  }
  // check basic validity
  TiXmlElement* root = XMLDoc->RootElement();
  assert( root != NULL);
  if( root == NULL || root->Value() == NULL || strcmp( root->Value(), "WorldDataFile"))
  {
    // report an error
    PRINTF(1)("Specified XML File is not an orxonox world data file (WorldDataFile element missing)\n");
    delete XMLDoc;
    return (ErrorMessage){213,"Path not a WorldDataFile","GameWorld::load()"};
  }
  /* the whole loading process for the GameWorld */
  this->dataTank->loadData(root);
  this->dataXML = (TiXmlElement*)root->Clone();

  delete XMLDoc;
  this->releaseLoadScreen();
}


/**
 *  unload the data of this GameWorld
 */
ErrorMessage GameWorld::unloadData()
{
  delete this->shell;
  PRINTF(3)("GameWorld::~GameWorld() - unloading the current GameWorld\n");

  this->dataTank->unloadData();

  State::setCurrentStoryEntity(NULL);
  if (this->dataXML)
    delete this->dataXML;
}


/**
 *  starts the GameWorld
 */
bool GameWorld::start()
{
  this->isPaused = false;
  this->isRunning = true;

  this->run();
}


/**
 *  stops the world.
 */
bool GameWorld::stop()
{
  PRINTF(3)("GameWorld::stop() - got stop signal\n");
  this->isRunning = false;
}


/**
 *  pauses the game
 */
bool GameWorld::pause()
{
  this->isPaused = true;
}


/**
 *  ends the pause Phase
 */
bool GameWorld::resume()
{
  this->isPaused = false;
}


/**
 *  main loop of the world: executing all world relevant function
 *
 * in this loop we synchronize (if networked), handle input events, give the heart-beat to
 * all other member-entities of the world (tick to player, enemies etc.), checking for
 * collisions drawing everything to the screen.
 */
void GameWorld::run()
{
  /* start the music */
  if(this->dataTank->music != NULL)
    this->dataTank->music->playback();

  this->lastFrame = SDL_GetTicks ();
  PRINTF(3)("GameWorld::mainLoop() - Entering main loop\n");

  while( this->isRunning) /* @todo implement pause */
  {
    ++this->cycle;
    /* process intput */
    this->handleInput ();
    if( !this->isRunning)
      break;

    /* network synchronisation */
    this->synchronize ();
    /* process time */
    this->tick ();
    /* process collision */
    this->collide ();
    /* update the state */
    this->update ();
    /* draw everything */
    this->display ();
  }

  PRINTF(3)("GameWorld::mainLoop() - Exiting the main loop\n");
}


/**
 *  synchronize local data with remote data
*/
void GameWorld::synchronize ()
{}


/**
 *  run all input processing

   the command node is the central input event dispatcher. the node uses the even-queue from
   sdl and has its own event-passing-queue.
*/
void GameWorld::handleInput ()
{
  EventHandler::getInstance()->process();
}


/**
 *  ticks a WorldEntity list
 * @param entityList list of the WorldEntities
 * @param dt time passed since last frame
 */
void GameWorld::tick(std::list<WorldEntity*> entityList, float dt)
{
  std::list<WorldEntity*>::iterator entity, next;
  next = entityList.begin();
  while (next != entityList.end())
  {
    entity = next++;
    (*entity)->tick(dt);
  }
}

/**
 *  advance the timeline
 *
 * this calculates the time used to process one frame (with all input handling, drawing, etc)
 * the time is mesured in ms and passed to all world-entities and other classes that need
 * a heart-beat.
 */
void GameWorld::tick ()
{
  Uint32 currentFrame = SDL_GetTicks();

  if( !this->isPaused)
  {
    this->dt = currentFrame - this->lastFrame;

    /* limit the the frame rate to 100 frames per second (fps) */
    if( this->dt < 10)
    {
      /* the frame-rate is limited to 100 frames per second, all other things are for nothing. */
      //PRINTF(0)("fps = 1000 - frame rate is adjusted\n");
      SDL_Delay(10 - dt);
      this->dt = 10;
    }

    this->dtS = (float)this->dt / 1000.0f * this->speed;
    this->gameTime += this->dtS;

    this->tick(this->dataTank->objectManager->getObjectList(OM_DEAD_TICK), this->dtS);
    this->tick(this->dataTank->objectManager->getObjectList(OM_ENVIRON), this->dtS);
    this->tick(this->dataTank->objectManager->getObjectList(OM_COMMON), this->dtS);
    this->tick(this->dataTank->objectManager->getObjectList(OM_GROUP_00), this->dtS);
    this->tick(this->dataTank->objectManager->getObjectList(OM_GROUP_00_PROJ), this->dtS);
    this->tick(this->dataTank->objectManager->getObjectList(OM_GROUP_01), this->dtS);
    this->tick(this->dataTank->objectManager->getObjectList(OM_GROUP_01_PROJ), this->dtS);

    /* update tick the rest */
    this->dataTank->localCamera->tick(this->dtS);
    AnimationPlayer::getInstance()->tick(this->dtS);
    PhysicsEngine::getInstance()->tick(this->dtS);

    GraphicsEngine::getInstance()->tick(this->dtS);

    if( likely(this->dataTank->gameRule != NULL))
      this->dataTank->gameRule->tick(this->dtS);
  }
  this->lastFrame = currentFrame;
}


/**
 *  this function gives the world a consistant state
 *
 * after ticking (updating the world state) this will give a constistant
 * state to the whole system.
 */
void GameWorld::update()
{
  GraphicsEngine::getInstance()->update(this->dtS);
  PNode::getNullParent()->updateNode (this->dtS);
  SoundEngine::getInstance()->update();

  if (this->dataTank->music != NULL)
    this->dataTank->music->update();
}


/**
 * kicks the CDEngine to detect the collisions between the object groups in the world
 */
void GameWorld::collide()
{
  CDEngine::getInstance()->checkCollisions(this->dataTank->objectManager->getObjectList(OM_GROUP_00),
      this->dataTank->objectManager->getObjectList(OM_GROUP_01_PROJ));
  CDEngine::getInstance()->checkCollisions(this->dataTank->objectManager->getObjectList(OM_GROUP_01),
      this->dataTank->objectManager->getObjectList(OM_GROUP_00_PROJ));
  CDEngine::getInstance()->checkCollisions(this->dataTank->objectManager->getObjectList(OM_GROUP_00),
  this->dataTank->objectManager->getObjectList(OM_GROUP_01));

  CDEngine::getInstance()->checkCollisions(this->dataTank->objectManager->getObjectList(OM_GROUP_01),
      this->dataTank->objectManager->getObjectList(OM_COMMON));
}

/**
 *  render the current frame
 *
 * clear all buffers and draw the world
 */
void GameWorld::display ()
{
  // clear buffer
  glClear( GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  // draw world
  this->draw();
  // flip buffers
  GraphicsEngine::swapBuffers();
}


/**
 *  runs through all entities calling their draw() methods
 */
void GameWorld::draw ()
{
  GraphicsEngine* engine = GraphicsEngine::getInstance();

  // set camera
  this->dataTank->localCamera->apply ();

  /* draw all WorldEntiy groups */
  engine->draw(State::getObjectManager()->getObjectList(OM_ENVIRON_NOTICK));
  engine->draw(State::getObjectManager()->getObjectList(OM_ENVIRON));
  engine->draw(State::getObjectManager()->getObjectList(OM_COMMON));
  engine->draw(State::getObjectManager()->getObjectList(OM_GROUP_00));
  engine->draw(State::getObjectManager()->getObjectList(OM_GROUP_00_PROJ));
  engine->draw(State::getObjectManager()->getObjectList(OM_GROUP_01));
  engine->draw(State::getObjectManager()->getObjectList(OM_GROUP_01_PROJ));


  if( unlikely( this->showBV))
  {
    CDEngine* engine = CDEngine::getInstance();
    engine->drawBV(State::getObjectManager()->getObjectList(OM_ENVIRON_NOTICK));
    engine->drawBV(State::getObjectManager()->getObjectList(OM_ENVIRON));
    engine->drawBV(State::getObjectManager()->getObjectList(OM_COMMON));
    engine->drawBV(State::getObjectManager()->getObjectList(OM_GROUP_00));
    engine->drawBV(State::getObjectManager()->getObjectList(OM_GROUP_01));
    engine->drawBV(State::getObjectManager()->getObjectList(OM_GROUP_01_PROJ));
  }

  if( unlikely(this->showPNodes))
    PNode::getNullParent()->debugDraw(0);

  engine->draw();

  // draw the game ruls
  if( likely(this->dataTank->gameRule != NULL))
    this->dataTank->gameRule->draw();
}

/**
 *  shows the loading screen
 */
void GameWorld::displayLoadScreen ()
{
  PRINTF(3)("GameWorld::displayLoadScreen - start\n");
  this->dataTank->glmis = new GLMenuImageScreen();
  this->dataTank->glmis->setMaximum(8);
  PRINTF(3)("GameWorld::displayLoadScreen - end\n");
}


/**
 *  removes the loadscreen, and changes over to the game
 */
void GameWorld::releaseLoadScreen ()
{
  PRINTF(3)("GameWorld::releaseLoadScreen - start\n");
  this->dataTank->glmis->setValue(this->dataTank->glmis->getMaximum());
  PRINTF(3)("GameWorld::releaseLoadScreen - end\n");
}



/**
 * @brief toggles the PNode visibility in the world (drawn as boxes)
 */
void GameWorld::togglePNodeVisibility()
{
  if (State::getCurrentStoryEntity() != NULL &&
      State::getCurrentStoryEntity()->isA(CL_GAME_WORLD))
  {
    dynamic_cast<GameWorld*>(State::getCurrentStoryEntity())->showPNodes = !dynamic_cast<GameWorld*>(State::getCurrentStoryEntity())->showPNodes;
  }
  else
    PRINTF(2)("The Current Story entity is not a GameWorld\n");
};


/**
 * @brief toggles the bounding volume (BV) visibility
*/
void GameWorld::toggleBVVisibility()
{
  if (State::getCurrentStoryEntity() != NULL &&
      State::getCurrentStoryEntity()->isA(CL_GAME_WORLD))
  {
    dynamic_cast<GameWorld*>(State::getCurrentStoryEntity())->showBV = !dynamic_cast<GameWorld*>(State::getCurrentStoryEntity())->showBV;
  }
  else
    PRINTF(2)("The Current Story entity is not a GameWorld\n");
};

