/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Silvan Nellen
   co-programmer: Benjamin Knecht
*/


#include "playable.h"

#include "weapons/weapon_manager.h"
#include "event_handler.h"
#include "player.h"
#include "state.h"

#include "world_entities/projectiles/projectile.h"

#include "power_ups/weapon_power_up.h"
#include "power_ups/param_power_up.h"

#include "game_rules.h"

#include "dot_emitter.h"
#include "sprite_particles.h"


Playable::Playable()
{
  this->setClassID(CL_PLAYABLE, "Playable");
  PRINTF(4)("PLAYABLE INIT\n");

  this->toList(OM_GROUP_01);
  this->weaponMan = new WeaponManager(this);

  // the reference to the Current Player is NULL, because we dont have one at the beginning.
  this->currentPlayer = NULL;

  this->bFire = false;
  this->oldFlags = 0;

  this->setSynchronized(true);

  this->score = 0;
  this->oldScore = 0;


  this->emitter = new DotEmitter(100, 5, M_2_PI);
  this->emitter->setParent(this);
  this->emitter->setSpread(M_PI, M_PI);
  this->emitter->setEmissionRate(300.0);
  this->emitter->setEmissionVelocity(50.0);

  this->explosionParticles = new SpriteParticles(1000);
  this->explosionParticles->setName("LaserExplosionParticles");
  this->explosionParticles->setLifeSpan(.5, .3);
  this->explosionParticles->setRadius(0.0, 10.0);
  this->explosionParticles->setRadius(.5, 6.0);
  this->explosionParticles->setRadius(1.0, 3.0);
  this->explosionParticles->setColor(0.0, 1,1,0,.9);
  this->explosionParticles->setColor(0.5, .8,.8,0,.5);
  this->explosionParticles->setColor(1.0, .8,.8,.7,.0);
}



Playable::~Playable()
{
  delete this->weaponMan;

  // THE DERIVED CLASS MUST UNSUBSCRIBE THE PLAYER THROUGH
  // this->setPlayer(NULL);
  // IN ITS DESTRUCTOR.
  assert(this->currentPlayer == NULL);
}


void Playable::addWeapon(Weapon* weapon, int configID, int slotID)
{
  this->weaponMan->addWeapon(weapon, configID, slotID);

  this->weaponConfigChanged();
}


void Playable::removeWeapon(Weapon* weapon)
{
  this->weaponMan->removeWeapon(weapon);

    this->weaponConfigChanged();
}


void Playable::nextWeaponConfig()
{
  this->weaponMan->nextWeaponConfig();
    this->weaponConfigChanged();
}


void Playable::previousWeaponConfig()
{
  this->weaponMan->previousWeaponConfig();
    this->weaponConfigChanged();
}


void Playable::weaponConfigChanged()
{
  if (this->currentPlayer != NULL)
    this->currentPlayer->weaponConfigChanged();
}


/**
 * @brief helps us colliding Playables
 */
void Playable::collidesWith(WorldEntity* entity, const Vector& location)
{
  if (entity == collider)
    return;
  collider = entity;

  if (entity->isA(CL_PROJECTILE))
  {
    this->decreaseHealth(entity->getHealth() *(float)rand()/(float)RAND_MAX);
    // EXTREME HACK
    if (this->getHealth() <= 0.0f)
    {
      this->die();
    }
  }
}


void Playable::respawn()
{
  PRINTF(0)("Playable respawn\n");
  // only if this is the spaceship of the player
  if( this == State::getPlayer()->getPlayable())
    State::getGameRules()->onPlayerSpawn();

  this->setAbsCoor(0.0, 0.0, 0.0);

  if( this->getOwner()%2 == 0)
    this->toList(OM_GROUP_00);
  else
    this->toList(OM_GROUP_01);
}


void Playable::die()
{
  PRINTF(0)("Playable dies\n");
  // only if this is the spaceship of the player
  if (State::isOnline())
  {
  if( this == State::getPlayer()->getPlayable())
    State::getGameRules()->onPlayerDeath();

  this->toList(OM_DEAD);
  //.HACK: moves the entity to an unknown place far far away: in the future, GameRules will look for that
  this->setAbsCoor(-2000.0, -2000.0, -2000.0);

  //explosion hack
  this->emitter->setSystem(explosionParticles);
  this->setAbsCoor(0, 0, 0);
  this->emitter->setSystem(NULL);
  }
}


/**
 * subscribe to all events the controllable needs
 * @param player the player that shall controll this Playable
 */
bool Playable::setPlayer(Player* player)
{
  // if we already have a Player inside do nothing
  if (this->currentPlayer != NULL && player != NULL)
  {
    return false;
  }

  // eject the Player if player == NULL
  if (this->currentPlayer != NULL && player == NULL)
  {
    PRINTF(4)("Player gets ejected\n");

    // unsubscibe all events.
    EventHandler* evh = EventHandler::getInstance();
    std::list<int>::iterator ev;
    for (ev = this->events.begin(); ev != events.end(); ev++)
      evh->unsubscribe( ES_GAME, (*ev));

    // leave the entity
    this->leave();

    // eject the current Player.
    Player* ejectPlayer = this->currentPlayer;
    this->currentPlayer = NULL;
    // eject the Player.
    ejectPlayer->setPlayable(NULL);

    return true;
  }

  // get the new Player inside
  if (this->currentPlayer == NULL && player != NULL)
  {
    PRINTF(4)("New Player gets inside\n");
    this->currentPlayer = player;
    if (this->currentPlayer->getPlayable() != this)
      this->currentPlayer->setPlayable(this);

    /*EventHandler*/
    EventHandler* evh = EventHandler::getInstance();
    std::list<int>::iterator ev;
    for (ev = this->events.begin(); ev != events.end(); ev++)
      evh->subscribe(player, ES_GAME, (*ev));

    this->enter();
    return true;
  }

  return false;
}


bool Playable::pickup(PowerUp* powerUp)
{
  if(powerUp->isA(CL_WEAPON_POWER_UP)) {
    return dynamic_cast<WeaponPowerUp*>(powerUp)->process(this->getWeaponManager());
  }
  else if(powerUp->isA(CL_PARAM_POWER_UP)) {
    ParamPowerUp* ppu = dynamic_cast<ParamPowerUp*>(powerUp);
    switch(ppu->getType()) {
      case POWERUP_PARAM_HEALTH:
        this->increaseHealth(ppu->getValue());
        return true;
      case POWERUP_PARAM_MAX_HEALTH:
        this->increaseHealthMax(ppu->getValue());
        return true;
    }
  }
  return false;
}

/**
 * add an event to the event list of events this Playable can capture
 * @param eventType the Type of event to add
 */
void Playable::registerEvent(int eventType)
{
  this->events.push_back(eventType);

  if (this->currentPlayer != NULL)
    EventHandler::getInstance()->subscribe(this->currentPlayer, ES_GAME, eventType);
}

/**
 * remove an event to the event list this Playable can capture.
 * @param event the event to unregister.
 */
void Playable::unregisterEvent(int eventType)
{
  this->events.remove(eventType);

  if (this->currentPlayer != NULL)
    EventHandler::getInstance()->unsubscribe(ES_GAME, eventType);
}

/**
 * @brief ticks a Playable
 * @param dt: the passed time since the last Tick
 */
void Playable::tick(float dt)
{
  this->weaponMan->tick(dt);
  if (this->bFire)
    weaponMan->fire();
}


/**
 * @brief processes Playable events.
 * @param event the Captured Event.
 */
void Playable::process(const Event &event)
{
  if( event.type == KeyMapper::PEV_FIRE1)
    this->bFire = event.bPressed;
  else if( event.type == KeyMapper::PEV_NEXT_WEAPON && event.bPressed)
  {
    this->nextWeaponConfig();
  }
  else if ( event.type == KeyMapper::PEV_PREVIOUS_WEAPON && event.bPressed)
    this->previousWeaponConfig();
}



void  Playable::attachCamera()
{
  State::getCameraNode()->setParentSoft(this);
  State::getCameraTargetNode()->setParentSoft(this);

}




void  Playable::detachCamera()
{
}

#define DATA_FLAGS    1
#define DATA_SCORE    2

#define FLAGS_bFire   1

int Playable::writeSync( const byte * data, int length, int sender )
{
  SYNCHELP_READ_BEGIN();

  byte b;
  SYNCHELP_READ_BYTE( b, NWT_PL_B );

  byte flags;

  if ( b == DATA_FLAGS )
  {
    SYNCHELP_READ_BYTE( flags, NWT_PL_FLAGS );

    bFire = (flags & FLAGS_bFire) != 0;

    return SYNCHELP_READ_N;
  }

  if ( b == DATA_SCORE )
  {
    int newScore;
    SYNCHELP_READ_BYTE( newScore, NWT_PL_SCORE );
    setScore( newScore );

    return SYNCHELP_READ_N;
  }

  return SYNCHELP_READ_N;
}

int Playable::readSync( byte * data, int maxLength )
{
  SYNCHELP_WRITE_BEGIN();

  if ( score != oldScore && isServer() )
  {
    SYNCHELP_WRITE_BYTE( DATA_SCORE, NWT_PL_B);
    SYNCHELP_WRITE_INT( score, NWT_PL_SCORE );
    oldScore = score;

    return SYNCHELP_WRITE_N;
  }

  byte flags = 0;

  if ( bFire )
    flags |= FLAGS_bFire;


  SYNCHELP_WRITE_BYTE( DATA_FLAGS, NWT_PL_B);
  SYNCHELP_WRITE_BYTE( flags, NWT_PL_FLAGS );
  oldFlags = flags;


  return SYNCHELP_WRITE_N;
}

bool Playable::needsReadSync( )
{
  if ( score != oldScore && isServer() )
    return true;

  byte flags = 0;

  if ( bFire )
    flags |= FLAGS_bFire;

  return flags!=oldFlags;
}
