/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli
*/


#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD

#include "game_world_data.h"

#include "resource_manager.h"
#include "state.h"
#include "class_list.h"
#include "substring.h"

#include "game_loader.h"

#include "p_node.h"
#include "world_entity.h"
#include "player.h"
#include "camera.h"
#include "environment.h"
#include "terrain.h"
#include "test_entity.h"
#include "terrain.h"
#include "skybox.h"
#include "md2Model.h"
#include "world_entities/projectiles/projectile.h"
#include "npcs/npc_test1.h"
#include "playable.h"

#include "light.h"

#include "factory.h"
#include "fast_factory.h"
#include "load_param.h"

#include "graphics_engine.h"
#include "event_handler.h"
#include "sound_engine.h"
#include "cd_engine.h"
#include "network_manager.h"
#include "physics_engine.h"
#include "fields.h"

#include "glmenu_imagescreen.h"

#include "animation_player.h"
#include "animation3d.h"

#include "game_rules.h"

#include "ogg_player.h"
#include "shader.h"


using namespace std;


/**
 * constructor of the GameWorldData
 */
GameWorldData::GameWorldData()
{
  this->setClassID(CL_GAME_WORLD_DATA, "GameWorldData");

  this->glmis = NULL;

  this->localCamera = NULL;
  this->localPlayer = NULL;
  this->sky = NULL;
  this->terrain = NULL;

  this->music = NULL;
  this->objectManager = NULL;
}


/**
 * destructor for the GameWorldData
 */
GameWorldData::~GameWorldData()
{}



/**
 *  initialize the GameWorldData
 */
ErrorMessage GameWorldData::init()
{
  this->objectManager = new ObjectManager();
  State::setObjectManager(this->objectManager);

  PNode::getNullParent();
  this->localCamera = new Camera();
  this->localCamera->setName ("GameWorld-Camera");
  State::setCamera(this->localCamera, this->localCamera->getTarget());

  LightManager::getInstance();

  GraphicsEngine::getInstance()->displayFPS(true);

  /* initialize some graphics engines and graphical elements */
  AnimationPlayer::getInstance();
  PhysicsEngine::getInstance();
}


/**
 *  loads the data from the xml file
 * @param root reference to the xml root element
 */
ErrorMessage GameWorldData::loadData(TiXmlElement* root)
{
  // load the parameters
  // name
  const char* string = grabParameter( root, "name");
  if( string == NULL)
  {
    PRINTF(2)("GameWorld is missing a proper 'name'\n");
    this->setName("Unknown");
  }
  else
    this->setName(string);

  this->loadGUI(root);
  this->loadWorldEntities(root);
  this->loadScene(root);
}


/**
 *  unloads the data from the xml file
 */
ErrorMessage GameWorldData::unloadData()
{
  this->unloadGUI();
  this->unloadWorldEntities();
  this->unloadScene();
}


/**
 *  loads the GUI data
 * @param root reference to the xml root element
 */
ErrorMessage GameWorldData::loadGUI(TiXmlElement* root)
{
  TiXmlElement* element = root->FirstChildElement("LoadScreen");
  if( element == NULL)
  {
    PRINTF(2)("no LoadScreen specified, loading default\n");

    glmis->setBackgroundImage("pictures/load_screen.jpg");
    this->glmis->setMaximum(8);
    //     this->glmis->draw();
  }
  else
  {
    this->glmis->loadParams(element);
    //     this->glmis->draw();
  }
  this->glmis->draw();
}


/**
 *  unloads the GUI data
 */
ErrorMessage GameWorldData::unloadGUI()
{
  delete this->glmis;
}


/**
 *  loads the world entities from the xml file
 * @param root reference to the xml root parameter
 */
ErrorMessage GameWorldData::loadWorldEntities(TiXmlElement* root)
{
  TiXmlElement* element = root->FirstChildElement("WorldEntities");

  if( element == NULL)
  {
    PRINTF(1)("GameWorld is missing 'WorldEntities'\n");
  }
  else
  {
    element = element->FirstChildElement();
    // load Players/Objects/Whatever
    PRINTF(4)("Loading WorldEntities\n");
    while( element != NULL)
    {
      BaseObject* created = Factory::fabricate(element);
      if( created != NULL )
        PRINTF(4)("Created a %s: %s\n", created->getClassName(), created->getName());

      //todo do this more elegant
      if( element->Value() != NULL && !strcmp( element->Value(), "SkyBox"))
      {
        this->sky = dynamic_cast<WorldEntity*>(created);
        State::setSkyBox(dynamic_cast<SkyBox*>(this->sky));
      }
      if( element->Value() != NULL && !strcmp( element->Value(), "Terrain"))
      {
        this->terrain = dynamic_cast<Terrain*>(created);
        CDEngine::getInstance()->setTerrain(terrain);
      }
      element = element->NextSiblingElement();
      this->glmis->step(); //! @todo temporary
    }
    PRINTF(4)("Done loading WorldEntities\n");
  }

  // Create a Player
  this->localPlayer = new Player();

  Playable* playable;
  const list<BaseObject*>* playableList = ClassList::getList(CL_PLAYABLE);
  if (playableList != NULL)
  {
    playable = dynamic_cast<Playable*>(playableList->front());
    this->localPlayer->setPlayable(playable);
  }

  /* init the pnode tree */
  PNode::getNullParent()->init();
}


/**
 *  unloads the world entities
 */
ErrorMessage GameWorldData::unloadWorldEntities()
{
  FastFactory::flushAll(true);
  GraphicsEngine::getInstance()->displayFPS(false);

  // erease everything that is left.
  // delete PNode::getNullParent(); // not needed as this is also done in the next step (and also much cleaner)
  const std::list<BaseObject*>* nodeList;
  //secondary cleanup of PNodes;
  nodeList = ClassList::getList(CL_PARENT_NODE);
  if (nodeList != NULL)
    while (!nodeList->empty())
      delete nodeList->front();
  /* remove the player object */
  if( this->localPlayer)
    delete this->localPlayer;

  nodeList = ClassList::getList(CL_GRAPHICS_EFFECT);
  if (nodeList != NULL)
    while (!nodeList->empty())
      delete nodeList->front();


  nodeList = ClassList::getList(CL_ELEMENT_2D);
    if (nodeList != NULL)
       while (!nodeList->empty())
         delete nodeList->front();




  // unload the resources !!
  ResourceManager::getInstance()->unloadAllByPriority(RP_LEVEL);

  if (State::getObjectManager() == this->objectManager)
  {
    State::setObjectManager(NULL);
    delete this->objectManager;
  }

  if(State::getSkyBox())
    State::setSkyBox(NULL);

  glmis = NULL;
  localCamera = NULL;
  localPlayer = NULL;
  sky = NULL;
  terrain = NULL;
  objectManager = NULL;
}


/**
 *  loads the scene data
 * @param root reference to the xml root element
 */
ErrorMessage GameWorldData::loadScene(TiXmlElement* root)
{
  LoadParamXML(root, "LightManager", LightManager::getInstance(), LightManager, loadParams);
  LoadParamXML(root, "GraphicsEngine", GraphicsEngine::getInstance(), GraphicsEngine, loadParams);

  LoadParam(root, "Music", this, GameWorldData, setSoundTrack);


  LoadParamXML(root, "GameRule", this, GameWorldData, loadGameRule);


  //LoadParamXML(root, "ParticleEngine", ParticleEngine::getInstance(), ParticleEngine, loadParams);
  //LoadParamXML(root, "PhysicsEngine", PhysicsEngine::getInstance(), PhysicsEngine, loadParams);

  this->localCamera->setClipRegion(1, 10000.0);
  if( this->sky != NULL)
    this->localCamera->addChild(this->sky);
  SoundEngine::getInstance()->setListener(this->localCamera);
}



/**
 *  unloads the scene data
 */
ErrorMessage GameWorldData::unloadScene()
{
  /* delete some garphics and scene eingines */
  delete LightManager::getInstance();
  delete AnimationPlayer::getInstance();
  delete PhysicsEngine::getInstance();

  if (this->music != NULL)
    this->setSoundTrack(NULL);
  this->music = NULL;
  /* stop the sound eninge */
  SoundEngine::getInstance()->flushAllBuffers();
  SoundEngine::getInstance()->flushAllSources();

  /* unload the shaders */
  Shader::suspendShader();
}


void GameWorldData::setSoundTrack(const char* name)
{
  if (this->music != NULL)
    delete this->music;
  this->music = NULL;

  if (name != NULL)
  {
    PRINTF(3)("Setting Sound Track to %s\n", name);
    char* oggFile = ResourceManager::getFullName(name); /// FIXME
    this->music = new OggPlayer(oggFile);
    delete[] oggFile;

    //(OggPlayer*)ResourceManager::getInstance()->load(name, OGG, RP_LEVEL);
    //assert(this->music->isA(CL_SOUND_OGG_PLAYER));
  }
}


void GameWorldData::loadGameRule(const TiXmlElement* root)
{

  const TiXmlElement* element = root->FirstChildElement("GameRule");

  if( element == NULL)
  {
    PRINTF(1)("GameWorld is missing 'GameRule'\n");
  }
  else
  {
    element = element->FirstChildElement();

    while( element != NULL)
    {
      BaseObject* created = Factory::fabricate(element);
      if (created == NULL /*|| !created->isA(CL_GAME_RULE)*/)
        delete created;
      else
      {
        this->gameRule = dynamic_cast<GameRules*>(created);
        element = element->NextSiblingElement();
      }
    }
  }



}



