/*!
 * @file particle_system.h

*/

#ifndef _PARTICLE_SYSTEM_H
#define _PARTICLE_SYSTEM_H

#include "world_entity.h"
#include "physics_interface.h"

#include "glincl.h"
#include "vector.h"
#include <list>

#include "quick_animation.h"

// Forward Declaration
class TiXmlElement;

#define PARTICLE_DOT_MASK              0x000001     //!< A Mask if the Particles should be displayed as DOTs
#define PARTICLE_SPARK_MASK            0x000010     //!< A Mask if the Particles should be displayed as SPARKs
#define PARTICLE_SPRITE_MASK           0x000100     //!< A Mask if the Particles should be displayed as SPRITESs
#define PARTICLE_MODEL_MASK            0x001000     //!< A Mask if the Particles should be displayed as MODELSs
#define PARTICLE_WORDL_ENTITY_MASK     0x010000     //!< A Mask if the Particles should be displayed as WORLD_ENTITIEs
#define PARTICLE_MULTI_MASK            0x100000     //!< A Mask if they are Multi-partilces

#define PARTICLE_DEFAULT_MAX_COUNT    200               //!< A default count of particles in the system.

// FORWARD DECLARATION
class Material;
class ParticleEmitter;
class Field;

//! A struct for one Particle
typedef struct Particle
{
  float         lifeTime;            //!< The time this particle has to live.
  float         lifeCycle;           //!< The fraction of time passed. (in percentage of its lifeTime)

  Vector        position;            //!< The current position of this particle.
  Vector        velocity;            //!< The current velocity of this Particle.
  Vector        extForce;            //!< The external Force that influences this Particle.
  Quaternion    orientation;         //!< The current orientation of this Particle.
  Quaternion    momentum;            //!< The current angular momentum (spin) of this Particle.
  float         mass;                //!< The mass of this Particle.
  float         massRand;            //!< A random mass
  float         radius;              //!< The current size of this Particle.
  float         radiusRand;          //!< a random Radius
  GLfloat       color [4];           //!< A Color for the particles.

  Particle*     next;                //!< pointer to the next particle in the List. (NULL if no preceding one)
};

//! A class to handle ParticleSystems
class ParticleSystem : public WorldEntity, public PhysicsInterface {

 public:
  ParticleSystem(unsigned int maxCount = PARTICLE_DEFAULT_MAX_COUNT);
  virtual ~ParticleSystem();

  void init();
  virtual void loadParams(const TiXmlElement* root);
  void loadEmitters(const TiXmlElement* root);

  void setLifeSpan(float lifeSpan, float randomLifeSpan = 0.0);
  void setConserve(float conserve);
  void setMaxCount(int maxCount);

  /* Per-Particle-Attributes */
  void setRadius(float lifeCycleTime, float radius, float randRadius = 0.0);
  void setMass(float lifeCycleTime, float mass, float randMass = 0.0);
  void setColor(float lifeCycleTime, float red, float green, float blue, float alpha);

  /** @returns the lifespan of the particles */
  inline float getLifeSpan() const { return this->lifeSpan; };
  /** @returns the starting-radius of the particles */
  inline float getStartRadius() { return this->radiusAnim.getValue(0.0); };
  /** @returns the end-radius of the particles */
  inline float getEndRadius() { return this->radiusAnim.getValue(1.0); };
  /** @returns the conserve-factor of the particles */
  inline float getConserve() const { return this->conserve; };
  /** @returns the initial mass of the particles */
  inline float getMass() const { return this->initialMass; };

  /** @returns the count of particles in this System */
  inline unsigned int getCount() const { return this->count; };
  /** @returns the maximum count of particles that can be contained by this System */
  inline unsigned int getMaxCount() const { return this->maxCount; };

  virtual unsigned int getFaceCount() const;

  void addEmitter(ParticleEmitter* emitter);
  void removeEmitter(ParticleEmitter* emitter);

  virtual void applyField(Field* field);
  /** @brief this is an empty function, because the Physics are implemented in tick @param dt: useless here */
  virtual void tickPhys(float dt) {};

  void addParticle(const Vector& position, const Vector& velocity, const Quaternion& orientation, const Quaternion& momentum, unsigned int data = 0);

  void precache(unsigned int seconds, unsigned int ticksPerSecond = 25);

  virtual void tick(float dt);
  virtual void draw() const = 0;

  void debug() const;

 protected:
  float             conserve;            //!< How much energy gets conserved to the next Tick.
  float             lifeSpan;            //!< Initial lifetime of a Particle.
  float             randomLifeSpan;      //!< A random value for the Lifespan (around the initial lifetime)
  float             initialMass;         //!< The initial Mass of the Particle
  float             randomInitialMass;   //!< The random initial Mass of the Particle

  int               maxCount;            //!< The maximum count of Particles.
  int               count;               //!< The current count of Particles.
  Particle*         particles;           //!< A list of particles of this System.
  Particle*         deadList;            //!< A list of dead Particles in the System.

  std::list<ParticleEmitter*> emitters;  //!< The Emitters that do emit into this System.

  // per particle attributes
  QuickAnimation    radiusAnim;          //!< Animation of the radius
  QuickAnimation    randRadiusAnim;      //!< Animation of the random Value of the radius
  QuickAnimation    massAnim;            //!< Animation of the mass
  QuickAnimation    randMassAnim;        //!< Animation of the random Mass
  QuickAnimation    colorAnim[4];        //!< Animation of the 4 colors (r,g,b,a)
};

#endif /* _PARTICLE_SYSTEM_H */
