/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD


#include "simple_game_menu.h"

#include "state.h"
#include "class_list.h"

#include "load_param.h"
#include "fast_factory.h"
#include "factory.h"

#include "p_node.h"
#include "world_entity.h"
#include "image_entity.h"
#include "terrain.h"
#include "camera.h"

#include "event_handler.h"
#include "graphics_engine.h"
#include "object_manager.h"


#include "cd_engine.h"


using namespace std;


//! This creates a Factory to fabricate a SimpleGameMenu
CREATE_FACTORY(SimpleGameMenu, CL_SIMPLE_GAME_MENU);



SimpleGameMenu::SimpleGameMenu(const TiXmlElement* root)
  : GameWorld(root)
{
  this->setClassID(CL_SIMPLE_GAME_MENU, "SimpleGameMenu");
  this->setName("SimpleGameMenu uninitialized");

  this->dataTank = new SimpleGameMenuData();

  this->cameraVector = Vector(50.0, 0.0, 0.0);
  this->menuLayer.push_back(new MenuLayer());
  this->menuLayer.push_back(new MenuLayer());
  this->layerIndex = 0;

  this->loadParams(root);
}


/**
 *  remove the SimpleGameMenu from memory
 *
 *  delete everything explicitly, that isn't contained in the parenting tree!
 *  things contained in the tree are deleted automaticaly
 */
SimpleGameMenu::~SimpleGameMenu ()
{
  PRINTF(3)("SimpleGameMenu::~SimpleGameMenu() - deleting current world\n");

  if( this->dataTank)
    delete this->dataTank;
}


/**
 * loads the parameters of a SimpleGameMenu from an XML-element
 * @param root the XML-element to load from
 */
void SimpleGameMenu::loadParams(const TiXmlElement* root)
{
  /* skip the GameWorld, since it does not define any useful loadParams for this class */
  //static_cast<GameWorld*>(this)->loadParams(root);
  GameWorld::loadParams(root);

  PRINTF(4)("Loaded SimpleGameMenu specific stuff\n");
}


/**
 * this is executed just before load
 *
 * since the load function sometimes needs data, that has been initialized
 * before the load and after the proceeding storyentity has finished
 */
ErrorMessage SimpleGameMenu::init()
{
  /* call underlying init funciton */
  GameWorld::init();

  EventHandler::getInstance()->subscribe(this, ES_MENU, SDLK_UP);
  EventHandler::getInstance()->subscribe(this, ES_MENU, SDLK_DOWN);
  EventHandler::getInstance()->subscribe(this, ES_MENU, SDLK_RETURN);
  EventHandler::getInstance()->subscribe(this, ES_MENU, SDLK_SPACE);

  this->dataTank->localCamera->setRelCoor(this->cameraVector);

  GraphicsEngine::getInstance()->displayFPS(false);
}


/**
 * load the data
 */
ErrorMessage SimpleGameMenu::loadData()
{
  GameWorld::loadData();

  if (this->dataXML != NULL)
  {
    TiXmlElement* element = this->dataXML->FirstChildElement("Elements");

    if( element == NULL)
    {
      PRINTF(1)("SimpleGameMenu is missing 'Elements'\n");
    }
    else
    {
      element = element->FirstChildElement();
    // load Players/Objects/Whatever
      PRINTF(4)("Loading Elements\n");
      while( element != NULL)
      {
        BaseObject* created = Factory::fabricate(element);
        if( created != NULL )
          PRINTF(4)("Created a %s: %s\n", created->getClassName(), created->getName());

      }
      PRINTF(4)("Done loading Elements\n");
    }
  }

  /* get the menu list */
  const std::list<BaseObject*>* imageEntityList = ClassList::getList(CL_IMAGE_ENTITY);
  std::list<BaseObject*>::const_iterator entity;
  for (entity = imageEntityList->begin(); entity != imageEntityList->end(); entity++)
  {

    if( !strcmp("Selector_Menu", (*entity)->getName()))
    {
      this->menuSelector = dynamic_cast<ImageEntity*>(*entity);
    }
    else if( !strcmp( "StartGame_Menu", (*entity)->getName()))
    {
      this->menuStartGame = dynamic_cast<ImageEntity*>(*entity);
      this->menuLayer[0]->menuList.push_back(dynamic_cast<ImageEntity*>(*entity));

    }
    else if( !strcmp( "Multiplayer_Menu", (*entity)->getName()))
    {
      this->menuStartMultiplayerGame = dynamic_cast<ImageEntity*>(*entity);
      this->menuLayer[0]->menuList.push_back(dynamic_cast<ImageEntity*>(*entity));
    }
    else if( !strcmp( "Quit_Menu", (*entity)->getName()))
    {
      this->menuQuitGame = dynamic_cast<ImageEntity*>(*entity);
      this->menuLayer[0]->menuList.push_back(dynamic_cast<ImageEntity*>(*entity));
    }
  }
  this->menuSelectedIndex = 0;
  this->menuSelected = this->menuLayer[0]->menuList[this->menuSelectedIndex];
  this->menuSelector->setAbsCoor(this->menuSelected->getAbsCoor());


  // loading the storyentities submenu (singleplayer)
  const std::list<BaseObject*>* storyEntities = ClassList::getList(CL_STORY_ENTITY);
  std::list<BaseObject*>::const_iterator it;
  for( it = storyEntities->begin(); it != storyEntities->end(); it++)
  {
    StoryEntity* se = dynamic_cast<StoryEntity*>(*it);
    if( se->isContainedInMenu())
    {
      this->menuLayer[1]->storyList.push_back(se);
      PRINTF(0)("Got a new menu entry |%s|\n", se->getName());
      ImageEntity* ie = new ImageEntity();
      ie->setAbsCoor(0.0f, this->menuLayer[1]->menuList.size() * -10.0f, 0.0f);
      //State::getObjectManager()->toList(dynamic_cast<WorldEntity*>(ie), OM_DEAD);
      //this->menuLayer[1]->menuList.push_back(ie);
    }
  }
}



/**
 * start the menu
 */
bool SimpleGameMenu::start()
{
  EventHandler::getInstance()->pushState(ES_MENU);

  /* now call the underlying*/
  GameWorld::start();
}



/**
 * stop the menu
 */
bool SimpleGameMenu::stop()
{
  EventHandler::getInstance()->popState();

  /* now call the underlying*/
  GameWorld::stop();
}


/**
 *  override the standard tick for more functionality
 */
void SimpleGameMenu::tick()
{
  GameWorld::tick();

  this->animateScene(this->dt);
}


/**
 *  no collision detection in the menu
 */
void SimpleGameMenu::collide()
{
//   this->dataTank->localCamera->
}


/**
 *  animate the scene
 */
void SimpleGameMenu::animateScene(float dt)
{
  Quaternion q(/*0.00005*/ 0.0001* dt, Vector(0.0, 1.0, 0.0));
  this->cameraVector = q.apply(this->cameraVector);
  this->dataTank->localCamera->setRelCoor(this->cameraVector);
  this->dataTank->localCamera->getTarget()->setRelCoorSoft(0,0,0);
}


/**
 * event dispatcher funciton
 * @param event the incoming event
 */
void SimpleGameMenu::process(const Event &event)
{
  /* ----------------- LAYER 1 ---------------*/
  if( this->layerIndex == 0)
  {
    if( event.type == SDLK_RETURN && event.bPressed == true)
    {
      if( this->menuSelected == this->menuQuitGame)
      {
        this->setNextStoryID(WORLD_ID_GAMEEND);
        this->stop();
      }
      if( this->menuSelected == this->menuStartGame)
      {
        //this->stop();
        // switch to first submenu
        if( this->menuLayer[1]->menuList.size() == 0)
        {
          PRINTF(1)("Haven't got any StoryEntities to play!\n");
          return;
        }

        this->switchMenuLayer(this->layerIndex, 1);

      }
    }
    else if( event.type == SDLK_DOWN && event.bPressed == true)
    {
//     ImageEntity*
      if(this->menuSelectedIndex < (this->menuLayer[this->layerIndex]->menuList.size() - 1))
      {
        this->menuSelected = this->menuLayer[this->layerIndex]->menuList[++this->menuSelectedIndex];
        this->menuSelector->setAbsCoor(this->menuSelected->getAbsCoor());
      }
    }
    else if( event.type == SDLK_UP && event.bPressed == true)
    {
      if(this->menuSelectedIndex > 0)
      {
        this->menuSelected = this->menuLayer[this->layerIndex]->menuList[--this->menuSelectedIndex];
        this->menuSelector->setAbsCoor(this->menuSelected->getAbsCoor());
      }
    }
  }  /* ----------------- LAYER 2 ---------------*/
  else if( this->layerIndex == 1)
  {

  }
}


/**
 *  switches to from one meny layer to an other
 * @param layer1 from layer
 * @param layer2 to layer
 */
void SimpleGameMenu::switchMenuLayer(int layer1, int layer2)
{
  // wrong sizes
  if(layer1 >= this->menuLayer.size() || layer2 >= this->menuLayer.size())
    return;


  PRINTF(0)("Removing layer %i\n", layer1);
  std::vector<ImageEntity*>::iterator it;
  // fade old menu
  for( it = this->menuLayer[layer1]->menuList.begin(); it != this->menuLayer[layer1]->menuList.end(); it++ )
  {
    (*it)->setAbsCoor(Vector(-100, -100, -100));
  }


  PRINTF(0)("Showing layer %i\n", layer1);
  // beam here the new menu
  for( it = this->menuLayer[layer2]->menuList.begin(); it != this->menuLayer[layer2]->menuList.end(); it++ )
  {}

  this->layerIndex = layer2;
}




/**********************************************************************************************
    SimpleGameMenuData
 **********************************************************************************************/


/**
 * SimpleGameMenuData constructor
 */
SimpleGameMenuData::SimpleGameMenuData()
{}

/**
 * SimpleGameMenuData decontructor
 */
SimpleGameMenuData::~SimpleGameMenuData()
{}


/**
 *  initialize the GameWorldDataData
 */
ErrorMessage SimpleGameMenuData::init()
{
  /* call underlying function */
  GameWorldData::init();
}


/**
 *  loads the GUI data
 * @param root reference to the xml root element
 */
ErrorMessage SimpleGameMenuData::loadGUI(TiXmlElement* root)
{
  /* call underlying function */
  GameWorldData::loadGUI(root);
}


/**
 *  unloads the GUI data
 */
ErrorMessage SimpleGameMenuData::unloadGUI()
{
  /* call underlying function */
  GameWorldData::unloadGUI();
}


/**
 *  overloads the GameWorld::loadWorldEntities(...) class since the menu WorldEntity loading is different (less loading stuff)
 * @param root reference to the xml root parameter
 */
ErrorMessage SimpleGameMenuData::loadWorldEntities(TiXmlElement* root)
{
  TiXmlElement* element = root->FirstChildElement("WorldEntities");

  if( element != NULL)
  {
    element = element->FirstChildElement();
    PRINTF(4)("Loading WorldEntities\n");
    while( element != NULL)
    {
      BaseObject* created = Factory::fabricate(element);
      if( created != NULL )
        printf("Created a %s: %s\n", created->getClassName(), created->getName());

      if( element->Value() != NULL && !strcmp( element->Value(), "SkyBox"))
        this->sky = dynamic_cast<WorldEntity*>(created);
      if( element->Value() != NULL && !strcmp( element->Value(), "Terrain"))
        this->terrain = dynamic_cast<Terrain*>(created);
      element = element->NextSiblingElement();
    }
    PRINTF(4)("Done loading WorldEntities\n");
  }

  /* init the pnode tree */
  PNode::getNullParent()->init();
}


/**
 *  unloads the world entities from the xml file
 */
ErrorMessage SimpleGameMenuData::unloadWorldEntities()
{
  /* call underlying function */
  GameWorldData::unloadWorldEntities();
}


/**
 *  loads the scene data
 * @param root reference to the xml root element
 */
ErrorMessage SimpleGameMenuData::loadScene(TiXmlElement* root)
{
  /* call underlying function */
  GameWorldData::loadScene(root);
}


/**
 *  unloads the scene data
 */
ErrorMessage SimpleGameMenuData::unloadScene()
{
  /* call underlying function */
  GameWorldData::unloadScene();
}



