/*!
 * @file game_world.h
 *  container for all game worlds (singleplayers, multiplayers..)
 */

#ifndef _GAME_WORLD_H
#define _GAME_WORLD_H


#include "story_entity.h"
#include "game_world_data.h"

class TiXmlElement;
class Shell;
class WorldEntity;
class GameWorldData;


//! The game world
/**
 *  this class initializes everything that should be displayed inside of the current level.
 *  it is the main driving factor during gameplay.
 */
class GameWorld : public StoryEntity
{

  public:
    GameWorld (const TiXmlElement* root = NULL);
    virtual ~GameWorld ();

    virtual void loadParams(const TiXmlElement* root);

    /* functions from story-entity */
    virtual ErrorMessage init();
    virtual ErrorMessage loadData();
    virtual ErrorMessage unloadData();

    virtual bool start();
    virtual bool stop();
    virtual bool pause();
    virtual bool resume();
    virtual void run();

    /**  this returns the current game time @returns elapsed game time     */
    inline double getGameTime() { return this->gameTime; }
    /** sets the game speed @param speed speed of the Game */
    inline void setSpeed(float speed) { this->speed = speed; };
    /**  returns the track path of this world @returns the track path */
    const char* getPath() { return this->path; }
    void setPath( const char* name);

    /** toggles the PNode visibility in the world (drawn as boxes) */
    void togglePNodeVisibility() { this->showPNodes = !this->showPNodes; };
    /** toggles the bounding volume (BV) visibility */
    void toggleBVVisibility() { this->showBV = !this->showBV; };




    inline void setSky(WorldEntity* sky) { this->dataTank->sky = sky; }


  protected:
    /* world - running functions */
    virtual void synchronize();
    virtual void handleInput();
    virtual void tick(std::list<WorldEntity*> worldEntity, float dt);
    virtual void tick();
    virtual void update();
    virtual void collide();
    virtual void draw();
    virtual void display();


  private:
    void displayLoadScreen();
    void releaseLoadScreen();


  protected:
    GameWorldData*      dataTank;                     //!< reference to the GameWorld Data Tank
    TiXmlElement*       dataXML;                      //!< The XML-Element this World has been loaded with.
    char*               path;                         //!< The file from which this world is loaded

    bool                showPNodes;                   //!< if the PNodes should be visible.
    bool                showBV;                       //!< if the Bounding Volumes should be visible.

    /* world timing */
    Uint32              lastFrame;                    //!< last time of frame
    Uint32              cycle;                        //!< The cycle we are in (starts with 0 and rises with every frame)
    Uint32              dt;                           //!< time needed to calculate this frame (in milliSeconds)
    float               dtS;                          //!< The time needed for caluculations in seconds
    float               speed;                        //!< how fast the game flows
    double              gameTime;                     //!< this is where the game time is saved

    /* external modules interfaces */
    Shell*              shell;
};

#endif /* _GAME_WORLD_H */
