/*!
 * @file story_entity.h
 *  holds the base class of everything that is playable - that is part of the story
 */


#ifndef _STORY_ENTITY_H
#define _STORY_ENTITY_H

#include "base_object.h"
#include "story_def.h"
#include "error.h"


typedef enum StoryEntityState
{
  SE_STATE_RUN     = 0,
  SE_STATE_STOP,
  SE_STATE_PAUSE,

  SE_STATE_NUM
};


//! A class that represents something to play in orxonox. it is a container for worlds, movies, mission briefings, etc...
class StoryEntity : virtual public BaseObject {

 public:
  StoryEntity ();
  virtual ~StoryEntity ();

  virtual void loadParams(const TiXmlElement* root);

  /* initialisation and loading */
  /** initializes a Story Entity to the needed values */
  virtual ErrorMessage init() {};
  /** called to load the data into the StoryEntity*/
  virtual ErrorMessage loadData() {};
  /** function that unloads the data from the StoryEntity */
  virtual ErrorMessage unloadData() {};

  /* running, stopping and pausing */
  /** starts the Entity. Starts the main cycle */
  virtual bool start() = 0;
  /**  Stops the entity. */
  virtual bool stop() = 0;
  /** pauses the Entity, you can resume the game by calling the resume() function */
  virtual bool pause() = 0;
  /** resumes a paused StoryEntity */
  virtual bool resume() = 0;
  /** function that is been called when the StoryEntity is started via start() */
  virtual void run() = 0;

  /* properties interface */
  /** returns the state of this StoryEntity */
  StoryEntityState getState();

  /** sets the story id of the current entity, this enables it to be identified in a global context.  @param storyID the story id */
  inline void setStoryID(int storyID) { this->storyID = storyID; }
  /** sets the story id of the current entity, this enables it to be identified in a  global context. @returns  the story id  */
  inline int getStoryID() { return this->storyID; }
  /**  sets the id of the next story entity: StoryEntities can choose their following entity themselfs.
   * the entity id defined here  will be startet after this entity ends. this can be convenient if you
   * want to have a non linear story with switches.
   * @param nextStoryID the story id of the next StoryEntity   */
  inline void setNextStoryID(int nextStoryID) { this->nextStoryID = nextStoryID; }
  /**  gets the story id of the current entity @returns story id */
  inline int getNextStoryID() { return this->nextStoryID; }
  /** sets the descroption of this StoryEntity @param name name */
  inline void setDescription(const char* description) { this->description = description; }
  /** @returns the description of this StoryEntity */
  inline const char* getDescription() { return this->description; }
  /** toggle the menu visibility: SimpleMenu specific */
  inline void addToGameMenu(int toggle) { this->bMenuEntry = (bool)toggle; }
  /** @returns true if the GameWorld should be contained in the SimpleMenu: SimpleMenu specific */
  inline bool isContainedInMenu() { return this->bMenuEntry; }
  /** sets the menu item image of this StoryEntity @param name name */
  inline void setMenuItemImage(const char* image) { this->menuItemImage = image; }
  /** @returns the menu item image of this StoryEntity */
  inline const char* getMenuItemImage() { return this->menuItemImage; }
  /** sets the menu screenshoot of this StoryEntity @param name name */
  inline void setMenuScreenshoot(const char* image) { this->menuScreenshoot = image; }
  /** @returns the menu screenshoot of this StoryEntity */
  inline const char* getMenuScreenshoot() { return this->menuScreenshoot; }


  protected:
    bool isInit;            //!< if the entity is initialized, this has to be true.
    bool isRunning;         //!< is true if the entity is running
    bool isPaused;          //!< is true if the entity is paused


 private:
    int storyID;            //!< this is the number of this entity, identifying it in a list/tree...
    int nextStoryID;        //!< if this entity has finished, this entity shall be called
    const char* description;//!< the description of the StoryEntity
    const char* menuItemImage;//!< the item image of the StoryEntity
    const char* menuScreenshoot;//!< the screenshoot of the StoryEntity
    bool        bMenuEntry; //!< If true, this GameWorld apears in the SimpleMenu: SimpleMenu specific
};

#endif /* _STORY_ENTITY_H */
