/*!
 * @file sound_engine.h
 * Definition of the SoundEngine singleton Class
 */

#ifndef _SOUND_ENGINE_H
#define _SOUND_ENGINE_H

#include "base_object.h"
#include "alincl.h"

#include "sound_buffer.h"
#include "sound_source.h"

#include <list>
#include <stack>

#define SOUND_DOPPLER_FACTOR       0.001          //!< A factor for the audible doppler effect
#define SOUND_DOPPLER_VELOCITY     5000000        //!< A factor for the TravelSpeed of sound

// FORWARD DECLARATION
class PNode;
class IniParser;


//! A class that handles audio via the openAudioLibrary
class SoundEngine : public BaseObject {
  public:
    virtual ~SoundEngine();
    /** @returns a Pointer to the only object of this Class */
    inline static SoundEngine* getInstance() { if (!SoundEngine::singletonRef) SoundEngine::singletonRef = new SoundEngine();  return SoundEngine::singletonRef; };

    void loadSettings(IniParser* iniParser);

    SoundSource* createSource(const char* fileName, PNode* sourceNode = NULL);

    /** @param listener the listener in the scene */
    void setListener(PNode* listener) { this->listener = listener; };
    void setDopplerValues(ALfloat dopplerFactor, ALfloat dopplerVelocity);


    /** @returns the Music Volume in % */
    inline float getMusicVolume() { return this->musicVolume; };
    /** @returns the Effects Volume in % */
    inline float getEffectsVolume() { return this->effectsVolume; };

    void update();

  // administrative
    void popALSource(ALuint& source);
    void pushALSource(ALuint& source) { if (source != 0) this->ALSources.push(source); };

    void flushUnusedBuffers();
    void flushAllBuffers();
    void flushAllSources();

    bool initAudio();
    bool allocateSources(unsigned int count);

  // error handling:
    static bool checkError(const char* error, unsigned int line);
    static const char* getALErrorString(ALenum err);

  private:
    SoundEngine();

    void listDevices();

  private:
    static SoundEngine*            singletonRef;             //!< Reference to this class

    ALCdevice*                     device;                   //!< the used audio-device.
    ALCcontext*                    context;                  //!< the context, currently in use.

    float                          musicVolume;              //!< the maximum volume of the music in % (0f,1f]
    float                          effectsVolume;            //!< the maximum volume of sound-effects in % (0f,1f]
    PNode*                         listener;                 //!< The listener of the Scene

    const std::list<BaseObject*>*  bufferList;               //!< A list of buffers
    const std::list<BaseObject*>*  sourceList;               //!< A list for all the sources in the scene.

    unsigned int                   maxSourceCount;           //!< How many Sources is the Maximum
    std::stack<ALuint>             ALSources;                //!< A list of real openAL-Sources, the engine allocates, and stores for reuse.
};

#endif /* _SOUND_ENGINE_H */
