/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Knecht
   co-programmer: Silvan Nellen

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "executor/executor.h"
#include "space_ship.h"

#include "objModel.h"
#include "resource_manager.h"

#include "weapons/weapon_manager.h"
#include "weapons/test_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"

#include "dot_emitter.h"
#include "sprite_particles.h"

#include "factory.h"
#include "key_mapper.h"
#include "event_handler.h"

#include "network_game_manager.h"

#include "power_ups/weapon_power_up.h"
#include "power_ups/param_power_up.h"

#include "graphics_engine.h"

#include "plane.h"

#include "state.h"
#include "player.h"


// #include "lib/gui/gl_gui/glgui_bar.h"
// #include "lib/gui/gl_gui/glgui_pushbutton.h"


using namespace std;

CREATE_FACTORY(SpaceShip, CL_SPACE_SHIP);


/**
 *  destructs the spaceship, deletes alocated memory
 */
SpaceShip::~SpaceShip ()
{
}

/**
 * loads a Spaceships information from a specified file.
 * @param fileName the name of the File to load the spaceship from (absolute path)
 */
SpaceShip::SpaceShip(const char* fileName)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for spaceship.\n", fileName);
    return;
  }

  this->loadParams(doc.RootElement());
}

/**
 *  creates a new Spaceship from Xml Data
 * @param root the xml element containing spaceship data

   @todo add more parameters to load
*/
SpaceShip::SpaceShip(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);
  else
  {
    this->loadModel("models/ships/reap_#.obj");
  }

}


/**
 * initializes a Spaceship
 */
void SpaceShip::init()
{
//  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->setClassID(CL_SPACE_SHIP, "SpaceShip");

  PRINTF(4)("SPACESHIP INIT\n");

  //weapons:
  Weapon* wpRight = new TestGun(0);
  wpRight->setName("testGun Right");
  Weapon* wpLeft = new TestGun(1);
  wpLeft->setName("testGun Left");
  Weapon* cannon = dynamic_cast<Weapon*>(Factory::fabricate(CL_CANNON));

  cannon->setName("BFG");

  this->addWeapon(wpLeft, 1, 0);
  this->addWeapon(wpRight,1 ,1);
  this->addWeapon(cannon, 0, 6);

  this->getWeaponManager()->changeWeaponConfig(1);

  bUp = bDown = bLeft = bRight = bAscend = bDescend = bRollL = bRollR = false;
  bFire = false;
  xMouse = yMouse = 0;
  yInvert = 1;
  mouseSensitivity = 0.001;
  airViscosity = 0.05;
  controlVelocityX = 25;
  controlVelocityY = 150;
  shipInertia = 1.5;
//  cycle = 0.0;

  this->setHealthMax(100);
  this->setHealth(80);

  travelSpeed = 40.0;
  acceleration = 3;
  this->velocity = this->getAbsDirX()*travelSpeed;
  this->mouseDir = this->getAbsDir();
  this->pitchDir = this->getAbsDir();

//   GLGuiButton* button = new GLGuiPushButton();
//    button->show();
//    button->setLabel("orxonox");
//    button->setBindNode(this);
//     GLGuiBar* bar = new GLGuiBar();
//     bar->show();
//     bar->setValue(7.0);
//     bar->setMaximum(10);
//     bar->setSize2D( 20, 100);
//     bar->setAbsCoor2D( 10, 200);

  //add events to the eventlist
  registerEvent(KeyMapper::PEV_UP);
  registerEvent(KeyMapper::PEV_DOWN);
  registerEvent(KeyMapper::PEV_LEFT);
  registerEvent(KeyMapper::PEV_RIGHT);
  //registerEvent(SDLK_q);
  //registerEvent(SDLK_e);
  registerEvent(KeyMapper::PEV_FIRE1);
  registerEvent(KeyMapper::PEV_NEXT_WEAPON);
  registerEvent(KeyMapper::PEV_PREVIOUS_WEAPON);
  //registerEvent(SDLK_PAGEUP);
  //registerEvent(SDLK_PAGEDOWN);
  registerEvent(EV_MOUSE_MOTION);

  this->getWeaponManager()->setSlotCount(7);

  this->getWeaponManager()->setSlotPosition(0, Vector(-2.6, .1, -3.0));
  this->getWeaponManager()->setSlotCapability(0, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager()->setSlotPosition(1, Vector(-2.6, .1, 3.0));
  this->getWeaponManager()->setSlotCapability(1, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager()->setSlotPosition(2, Vector(-1.5, .5, -.5));
  this->getWeaponManager()->setSlotDirection(2, Quaternion(-M_PI_4*.5, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(3, Vector(-1.5, .5, .5));
  this->getWeaponManager()->setSlotDirection(3, Quaternion(M_PI_4*.5, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(4, Vector(-1.5, -.5, .5));
  this->getWeaponManager()->setSlotDirection(4, Quaternion(-M_PI_4*.5+M_PI, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(5, Vector(-1.5, -.5, -.5));
  this->getWeaponManager()->setSlotDirection(5, Quaternion(+M_PI_4*.5-M_PI, Vector(1,0,0)));
//
   this->getWeaponManager()->setSlotPosition(6, Vector(-1, 0.0, 0));
   this->getWeaponManager()->setSlotCapability(6, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);
   //
//   this->getWeaponManager()->setSlotPosition(8, Vector(-2.5, -0.3, -2.0));
//   this->getWeaponManager()->setSlotDirection(8, Quaternion(-M_PI, Vector(1,0,0)));
//
//   this->getWeaponManager()->setSlotPosition(9, Vector(-2.5, -0.3, 2.0));
//   this->getWeaponManager()->setSlotDirection(9, Quaternion(+M_PI, Vector(1,0,0)));:

  this->getWeaponManager()->getFixedTarget()->setParent(this);
  this->getWeaponManager()->getFixedTarget()->setRelCoor(100000,0,0);

  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);

  this->burstEmitter = new DotEmitter(200, 0.0, .01);
  this->burstEmitter->setParent(this);
  this->burstEmitter->setRelCoor(-1, .5, 0);
  this->burstEmitter->setName("SpaceShip_Burst_emitter");

  this->burstSystem = new SpriteParticles(1000);
  this->burstSystem->addEmitter(this->burstEmitter);
  this->burstSystem->setName("SpaceShip_Burst_System");
  ((SpriteParticles*)this->burstSystem)->setMaterialTexture("maps/radial-trans-noise.png");
  this->burstSystem->setLifeSpan(1.0, .3);
  this->burstSystem->setRadius(0.0, 1.0);
  this->burstSystem->setRadius(0.05, 1.0);
  this->burstSystem->setRadius(.5, .8);
  this->burstSystem->setRadius(1.0, 0);
  this->burstSystem->setColor(0.0, .7,.7,1,.7);
  this->burstSystem->setColor(0.2, 0,0,0.8,.5);
  this->burstSystem->setColor(0.5, .5,.5,.8,.8);
  this->burstSystem->setColor(1.0, .8,.8,.8,.0);

}


/**
 * loads the Settings of a SpaceShip from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void SpaceShip::loadParams(const TiXmlElement* root)
{
  Playable::loadParams(root);
}


void SpaceShip::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( true);
  this->attachCamera();
}

void SpaceShip::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);
  this->detachCamera();
}


/**
 *  effect that occurs after the SpaceShip is spawned
*/
void SpaceShip::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}

/**
 *  the action occuring if the spaceship left the game
*/
void SpaceShip::leftWorld ()
{}

WorldEntity* ref = NULL;
/**
 *  this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void SpaceShip::collidesWith(WorldEntity* entity, const Vector& location)
{
  Playable::collidesWith(entity, location);
  if (entity->isA(CL_TURRET_POWER_UP) && entity != ref)
  {
    this->ADDWEAPON();
    ref = entity;
    }
//  PRINTF(3)("collision %s vs %s @ (%f,%f,%f)\n", this->getClassName(), entity->getClassName(), location.x, location.y, location.z);
}

/**
 *  draws the spaceship after transforming it.
*/
void SpaceShip::draw () const
{
  WorldEntity::draw();
  this->getWeaponManager()->draw();

  //this->debug(0);
}

/**
 *  the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void SpaceShip::tick (float time)
{
  Playable::tick(time);

  if( ( xMouse != 0 || yMouse != 0 ) && this->getOwner() == this->getHostID() )
   {
    if (xMouse > controlVelocityX) xMouse = controlVelocityX;
    else if (xMouse < -controlVelocityX) xMouse = -controlVelocityX;
    if (yMouse > controlVelocityY) yMouse = controlVelocityY;
    else if (yMouse < -controlVelocityY) yMouse = -controlVelocityY;

    pitchDir = (Quaternion(xMouse*mouseSensitivity*0.5, Vector(1,0,0)));

    mouseDir *= (Quaternion(-M_PI/4*xMouse*mouseSensitivity, Vector(0,1,0))*Quaternion(-M_PI/4*yMouse*mouseSensitivity*yInvert, Vector(0,0,1))*pitchDir);
    xMouse = yMouse = 0;
   }


//   if( this != State::getPlayer()->getControllable())
//     return;

  // spaceship controlled movement
  this->calculateVelocity(time);

  Vector move = velocity*time;

  //orient the velocity in the direction of the spaceship.
  travelSpeed = velocity.len();
  velocity += ((this->getAbsDirX())*travelSpeed-velocity)*airViscosity;
  velocity = (velocity.getNormalized())*travelSpeed;
  this->burstEmitter->setEmissionRate(travelSpeed);
  this->burstEmitter->setEmissionVelocity(travelSpeed*.5, travelSpeed *.1);

  //orient the spaceship in direction of the mouse
   rotQuat = Quaternion::quatSlerp( this->getAbsDir(), mouseDir, 0.5);//fabsf(time)*shipInertia);
   if (this->getAbsDir().distance(rotQuat) > 0.00000000000001)
    this->setAbsDir( rotQuat);
   //this->setAbsDirSoft(mouseDir,5);

  // this is the air friction (necessary for a smooth control)
  if(travelSpeed >= 120) velocity -= velocity.getNormalized()*travelSpeed*travelSpeed*0.0001;
  else if (travelSpeed <= 80) velocity -= velocity.getNormalized()*travelSpeed*0.001;

  //other physics (gravity)
  //if(travelSpeed < 120)
  //move += Vector(0,-1,0)*60*time + Vector(0,1,0)*travelSpeed/2*time;

  //hoover effect
  //cycle += time;
  //this->shiftCoor(Vector(0,1,0)*cos(this->cycle*2.0)*0.02);

  //readjust
  //if (this->getAbsDirZ().y > 0.1) this->shiftDir(Quaternion(time*0.3, Vector(1,0,0)));
  //else if (this->getAbsDirZ().y < -0.1) this->shiftDir(Quaternion(-time*0.3, Vector(1,0,0)));

  //SDL_WarpMouse(GraphicsEngine::getInstance()->getResolutionX()/2, GraphicsEngine::getInstance()->getResolutionY()/2);

  this->shiftCoor(move);


}

/**
 *  calculate the velocity
 * @param time the timeslice since the last frame
*/
void SpaceShip::calculateVelocity (float time)
{
  Vector accel(0.0, 0.0, 0.0);
  /*
  Vector rot(0.0, 0.0, 0.0); // wird bentigt fr Helicopter
  */
  //float rotVal = 0.0;
  /* FIXME: calculating the direction and orthDirection every timeSlice is redundant! save it somewhere */
  /* calculate the direction in which the craft is heading  */

  //Plane plane(Vector(0,1,0), Vector(0,0,0));

  if( this->bUp )
   {
     //this->shiftCoor(this->getAbsDirX());
      //accel += (this->getAbsDirX())*2;
      accel += (this->getAbsDirX())*acceleration;

   }

  if( this->bDown )
   {
     //this->shiftCoor((this->getAbsDirX())*-1);
     //accel -= (this->getAbsDirX())*2;
    //if(velocity.len() > 50)
     accel -= (this->getAbsDirX())*0.5*acceleration;



   }

  if( this->bLeft/* > -this->getRelCoor().z*2*/)
  {
    this->shiftDir(Quaternion(time, Vector(0,1,0)));
//    accel -= rightDirection;
    //velocityDir.normalize();
    //rot +=Vector(1,0,0);
    //rotVal -= .4;
  }
  if( this->bRight /* > this->getRelCoor().z*2*/)
  {
    this->shiftDir(Quaternion(-time, Vector(0,1,0)));

    //    accel += rightDirection;
    //velocityDir.normalize();
    //rot += Vector(1,0,0);
    //rotVal += .4;
  }


  if( this->bRollL /* > -this->getRelCoor().z*2*/)
  {
    mouseDir *= Quaternion(-time*2, Vector(1,0,0));
//    accel -= rightDirection;
    //velocityDir.normalize();
    //rot +=Vector(1,0,0);
    //rotVal -= .4;
  }
  if( this->bRollR /* > this->getRelCoor().z*2*/)
  {
    mouseDir *= Quaternion(time*2, Vector(1,0,0));

    //    accel += rightDirection;
    //velocityDir.normalize();
    //rot += Vector(1,0,0);
    //rotVal += .4;
  }
  if (this->bAscend )
  {
    this->shiftDir(Quaternion(time, Vector(0,0,1)));

//    accel += upDirection;
    //velocityDir.normalize();
    //rot += Vector(0,0,1);
    //rotVal += .4;
  }
  if (this->bDescend )
  {
    this->shiftDir(Quaternion(-time, Vector(0,0,1)));

    //    accel -= upDirection;
    //velocityDir.normalize();
    //rot += Vector(0,0,1);
    //rotVal -= .4;
  }

  velocity += accel;
  //rot.normalize();
  //this->setRelDirSoft(Quaternion(rotVal, rot), 5);
}

/**
 * @todo switch statement ??
 */
void SpaceShip::process(const Event &event)
{
  Playable::process(event);

  if( event.type == KeyMapper::PEV_LEFT)
      this->bRollL = event.bPressed;
  else if( event.type == KeyMapper::PEV_RIGHT)
      this->bRollR = event.bPressed;
  else if( event.type == KeyMapper::PEV_UP)
    this->bUp = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_DOWN)
    this->bDown = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == EV_MOUSE_MOTION)
  {
    this->xMouse += event.xRel;
    this->yMouse += event.yRel;
  }
}

#include "weapons/aiming_turret.h"
// FIXME THIS MIGHT BE CONSIDERED EITHER A FEATURE, OR A BUG
void SpaceShip::ADDWEAPON()
{
  Weapon* turret = NULL;

  if ((float)rand()/RAND_MAX < .9)
  {
    //if (this->getWeaponManager()->hasFreeSlot(2, WTYPE_TURRET))
    {
      turret = new Turret();
      this->addWeapon(turret, 2);
      this->getWeaponManager()->changeWeaponConfig(2);
    }
  }
  else
  {
    //if (this->getWeaponManager()->hasFreeSlot(3))
    {
      turret = dynamic_cast<Weapon*>(Factory::fabricate(CL_TARGETING_TURRET));
      if (turret != NULL)
      this->addWeapon(turret, 3);

      this->getWeaponManager()->changeWeaponConfig(3);
    }
  }

  if(turret != NULL)
  {
    turret->setName("Turret");
    turret->setStateDuration(WS_SHOOTING, (float)rand()/RAND_MAX*.5+.1);
  }
}

#define MASK_bUp         1
#define MASK_bDown       2
#define MASK_bLeft       4
#define MASK_bRight      8
#define MASK_bAscend    16
#define MASK_bDescend   32
#define MASK_bFire      64
#define MASK_bRollL    128
#define MASK_bRollR    256

#define DATA_state       1
#define DATA_flags       2
#define DATA_mouse       3
#define DATA_sync        4
#define DATA_velocity    5

int SpaceShip::writeBytes( const byte * data, int length, int sender )
{
  SYNCHELP_READ_BEGIN();

  byte b;

  while ( SYNCHELP_READ_REMAINING()>0 )
  {
    SYNCHELP_READ_BYTE( b, NWT_SS_B );

    if ( b == DATA_state )
    {
      setRequestedSync( false );
      setIsOutOfSync( false );
      SYNCHELP_READ_FKT( WorldEntity::writeState, NWT_SS_WE_STATE );

      continue;
    }

    if ( b == DATA_flags )
    {
      int flags = 0;
      SYNCHELP_READ_INT( flags, NWT_SS_FLAGS );

      bUp = (flags & MASK_bUp) != 0;
      bDown = (flags & MASK_bDown) != 0;
      bLeft = (flags & MASK_bLeft) != 0;
      bRight = (flags & MASK_bRight) != 0;
      bAscend = (flags & MASK_bAscend) != 0;
      bDescend = (flags & MASK_bDescend) != 0;
      bFire = (flags & MASK_bFire) != 0;
      bRollL = (flags & MASK_bRollL) != 0;
      bRollR = (flags & MASK_bRollR) != 0;

      continue;
    }

    if ( b == DATA_mouse )
    {
      SYNCHELP_READ_FLOAT( mouseDir.w, NWT_SS_MOUSEDIRW );
      SYNCHELP_READ_FLOAT( mouseDir.v.x, NWT_SS_MOUSEDIRX );
      SYNCHELP_READ_FLOAT( mouseDir.v.y, NWT_SS_MOUSEDIRY );
      SYNCHELP_READ_FLOAT( mouseDir.v.z, NWT_SS_MOUSEDIRZ );

      continue;
    }

    if ( b == DATA_sync )
    {
      if ( this->getOwner() != this->getHostID() )
        SYNCHELP_READ_FKT( PNode::writeSync, NWT_SS_PN_SYNC );

      continue;
    }

    if ( b == DATA_velocity )
    {
      SYNCHELP_READ_FLOAT( velocity.x, NWT_SS_VELX );
      SYNCHELP_READ_FLOAT( velocity.y, NWT_SS_VELY );
      SYNCHELP_READ_FLOAT( velocity.z, NWT_SS_VELZ );
    }
  }

  return SYNCHELP_READ_N;
}



int SpaceShip::readBytes( byte * data, int maxLength, int * reciever )
{
  SYNCHELP_WRITE_BEGIN();

  if ( isOutOfSync() && !requestedSync() /*&& this->getHostID()!=this->getOwner()*/ )
  {
    (NetworkGameManager::getInstance())->sync( this->getUniqueID(), this->getOwner() );
    setRequestedSync( true );
    PRINTF(0)("REQUESTED STATE %d\n", this->getUniqueID());
  }

  int rec = this->getRequestSync();
  if ( rec > 0 )
  {
    *reciever = rec;

    PRINTF(0)("SEND STATE %d %d\n", this->getUniqueID(), rec);

    SYNCHELP_WRITE_BYTE( (byte)DATA_state, NWT_SS_B );

    SYNCHELP_WRITE_FKT( WorldEntity::readState, NWT_SS_WE_STATE );
    //SYNCHELP_WRITE_FLOAT( cycle );

    return SYNCHELP_WRITE_N;
  }

  *reciever = 0;

  if ( this->getOwner() == this->getHostID() && PNode::needsReadSync() )
  {
    SYNCHELP_WRITE_BYTE( DATA_sync, NWT_SS_B );
    SYNCHELP_WRITE_FKT( PNode::readSync, NWT_SS_PN_SYNC );
  }

  if ( this->getHostID()==this->getOwner() )
  {
    int mask = 0;

    if ( bUp )
      mask |= MASK_bUp;
    if ( bDown )
      mask |= MASK_bDown;
    if ( bLeft )
      mask |= MASK_bLeft;
    if ( bRight )
      mask |= MASK_bRight;
    if ( bAscend )
      mask |= MASK_bAscend;
    if ( bFire )
      mask |= MASK_bFire;
    if ( bRollL )
      mask |= MASK_bRollL;
    if ( bRollR )
      mask |= MASK_bRollR;


    //static float oldxMouse = xMouse + 1.0;
    //static float oldyMouse = yMouse + 1.0;

    if ( mask != oldMask )
    {
      oldMask = mask;
      SYNCHELP_WRITE_BYTE( DATA_flags, NWT_SS_B );
      SYNCHELP_WRITE_INT( mask, NWT_SS_FLAGS );
    }
#define __OFFSET_MDIR_W 0.01
#define __OFFSET_MDIR_V 0.01
    if ( fabs( oldMouseDir.w - mouseDir.w ) > __OFFSET_MDIR_W ||
         fabs( oldMouseDir.v.x - mouseDir.v.x ) > __OFFSET_MDIR_V ||
         fabs( oldMouseDir.v.y - mouseDir.v.y ) > __OFFSET_MDIR_V ||
         fabs( oldMouseDir.v.z - mouseDir.v.z ) > __OFFSET_MDIR_V )
    {
      oldMouseDir = mouseDir;

      SYNCHELP_WRITE_BYTE( DATA_mouse, NWT_SS_B );
      PRINTF(0)("SENDING mousedir\n");
      SYNCHELP_WRITE_FLOAT( mouseDir.w, NWT_SS_MOUSEDIRW );
      SYNCHELP_WRITE_FLOAT( mouseDir.v.x, NWT_SS_MOUSEDIRX );
      SYNCHELP_WRITE_FLOAT( mouseDir.v.y, NWT_SS_MOUSEDIRY );
      SYNCHELP_WRITE_FLOAT( mouseDir.v.z, NWT_SS_MOUSEDIRZ );
    }
#define __OFFSET_VEL 0.05
    if ( fabs( oldVelocity.x - velocity.x ) > __OFFSET_VEL*velocity.x ||
         fabs( oldVelocity.y - velocity.y ) > __OFFSET_VEL*velocity.y ||
         fabs( oldVelocity.z - velocity.z ) > __OFFSET_VEL*velocity.z )
    {
      oldVelocity = velocity;
      PRINTF(0)("SENDING velocity\n");
      SYNCHELP_WRITE_BYTE( DATA_velocity, NWT_SS_B );
      SYNCHELP_WRITE_FLOAT( velocity.x, NWT_SS_VELX );
      SYNCHELP_WRITE_FLOAT( velocity.y, NWT_SS_VELY );
      SYNCHELP_WRITE_FLOAT( velocity.z, NWT_SS_VELZ );
    }

  }

  return SYNCHELP_WRITE_N;
}


