/*!
 * @file particle_emitter.h
  *  Definition of a ParticleEmitter
*/

#ifndef _PARTICLE_EMITTER_H
#define _PARTICLE_EMITTER_H

#include "p_node.h"

// FORWARD DECLARATION
class ParticleSystem;
class TiXmlElement;

// Default values
#define PARTICLE_EMITTER_DEFAULT_SIZE              1.0
#define PARTICLE_EMITTER_DEFAULT_EMISSION_RATE     50
#define PARTICLE_EMITTER_DEFAULT_INHERIT_SPEED     0.0
#define PARTICLE_EMITTER_DEFAULT_SPREAD            M_PI

//! A class to handle an Emitter.
class ParticleEmitter : public PNode
{
  friend class ParticleSystem;
public:
  ParticleEmitter(const Vector& direction = Vector(1.0,0.0,0.0) , float angle = .5,
                  float emissionRate = 1.0, float velocity = 1.0);
  virtual ~ParticleEmitter();

  virtual void loadParams(const TiXmlElement* root = NULL);

  /* controlling the emitter: interface */
  void start();
  void stop();
  void tick(float dt);

  void setSystem(ParticleSystem* system);
  ParticleSystem* getSystem() const { return this->system; };

  /* controlling the behavour: these can be used as Animation interfaces */
  void setEmissionRate(float emissionRate);
  void setInheritSpeed(float value);
  void setSpread(float angle, float randomAngle = 0.0);
  void setEmissionVelocity(float velocity, float randomVelocity = 0.0);
  void setEmissionMomentum(float momentum, float randomMomentum = 0.0);

  void setDirection(float x, float y, float z) { this->direction = Vector(x,y,z); }; //!< todo this should be done via PNODE

  /** @returns the emissionRate */
  inline float getEmissionRate() const { return this->emissionRate; };
  /** @returns the inherit-speed-factor */
  inline float getInheritSpeed() const { return this->inheritSpeed; };
  /** @returns the SpreadAngle of the emitter */
  inline float getSpread() const { return this->angle; };
  /** @returns the EmissionVelocity of the emitter */
  inline float getEmissionVelocity() const { return this->velocity; };
  /** @returns the EmissionMomentum of this emitter */
  inline float getEmissionMomentum() const { return this->momentum; };

  virtual void emitParticles(unsigned int count) const = 0;

  void debug() const;

protected:

  float           inheritSpeed;      //!< How much speed the particle inherits from the Emitters speed.
  Vector          direction;         //!< emition direction.
  float           angle;             //!< max angle from the direction of the emitter
  float           randomAngle;       //!< random emission angle (angle +- angleRandom is the emitted angle.
  float           velocity;          //!< the initial speed of a Particles.
  float           randomVelocity;    //!< the random variation from the initial Speed.
  float           momentum;          //!< The Initial spped of the Rotation.
  float           momentumRandom;    //!< The random variation of the Momentum.

private:
  ParticleSystem* system;            //!< The ParticleSystem this Emitter Emits into.
  float           saveTime;          //!< The time that was missing by the last Tick (otherwise there would be no emission when framefate is too big).
  float           emissionRate;      //!< amount of particles per seconds emitted by emitter.
};

#endif /* _PARTICLE_EMITTER_H */
