/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Patrick Boenzli
*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WEAPON

#include "billboard.h"

#include "load_param.h"
#include "factory.h"

#include "graphics_engine.h"
#include "material.h"
#include "glincl.h"
#include "state.h"
#include "p_node.h"


using namespace std;


CREATE_FACTORY(Billboard, CL_IMAGE_ENTITY);


/**
 * standart constructor
 */
Billboard::Billboard (const TiXmlElement* root)
{
  this->init();

  if( root)
    this->loadParams(root);
}


/**
 * destroys a Billboard
 */
Billboard::~Billboard ()
{
  if (this->material)
    delete this->material;
}


/**
 * initializes the Billboard
 */
void Billboard::init()
{
  this->setClassID(CL_BILLBOARD, "Billboard");
  this->setName("Billboard");

  this->setLayer(E2D_LAYER_TOP);
  this->setSize(GraphicsEngine::getInstance()->getResolutionX()/10.0, GraphicsEngine::getInstance()->getResolutionY()/10.0);

  //this->setBindNode(this);
  this->material = new Material();
  this->setTexture("pictures/error_texture.png");
}


/**
 *  load params
 * @param root TiXmlElement object
 */
void Billboard::loadParams(const TiXmlElement* root)
{
  LoadParam(root, "texture", this, Billboard, setTexture)
      .describe("the texture-file to load onto the Billboard");

  LoadParam(root, "size", this, Billboard, setSize)
      .describe("the size of the Billboard in Pixels");
}


/**
 * sets the size of the Billboard.
 * @param size the size in pixels
 */
void Billboard::setSize(float sizeX, float sizeY)
{
  this->setSize2D(sizeX, sizeY);
}


/**
 * sets the material to load
 * @param textureFile The texture-file to load onto the crosshair
 */
void Billboard::setTexture(const char* textureFile)
{
  this->material->setDiffuseMap(textureFile);
}


/**
 * attaches this billboard to a parent
 * @param pNode node to attach to
 */
void Billboard::attachTo(PNode* pNode)
{
  this->source->setParent(pNode);
}


/**
 * ticks the Billboard
 * @param dt the time to ticks
 */
void Billboard::tick(float dt)
{
  float z = 0.0f;
  glReadPixels ((int)this->getAbsCoor2D().x,
                 GraphicsEngine::getInstance()->getResolutionY()-(int)this->getAbsCoor2D().y-1,
                 1,
                 1,
                 GL_DEPTH_COMPONENT,
                 GL_FLOAT,
                 &z);

  GLdouble objX=.0, objY=.0, objZ=.0;
  gluUnProject(this->getAbsCoor2D().x,
               GraphicsEngine::getInstance()->getResolutionY()-this->getAbsCoor2D().y-1,
               .99,  // z
               GraphicsEngine::modMat,
               GraphicsEngine::projMat,
               GraphicsEngine::viewPort,
               &objX,
               &objY,
               &objZ );
}


/**
 * draws the billboard
 */
void Billboard::draw() const
{
  glPushMatrix();

  glTranslatef(this->getAbsCoor2D().x, this->getAbsCoor2D().y, 0);
  this->material->select();

  glBegin(GL_TRIANGLE_STRIP);
  glTexCoord2f(0, 0);
  glVertex2f(-this->getSizeX2D(), -this->getSizeY2D());
  glTexCoord2f(1, 0);
  glVertex2f(this->getSizeX2D(), -this->getSizeY2D());
  glTexCoord2f(0, 1);
  glVertex2f(-this->getSizeX2D(), this->getSizeY2D());
  glTexCoord2f(1, 1);
  glVertex2f(this->getSizeX2D(), this->getSizeY2D());
  glEnd();

  glPopMatrix();
}
