/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...

*/

#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY

#include "hover.h"

#include "weapons/weapon_manager.h"
#include "weapons/test_gun.h"
#include "weapons/turret.h"
#include "weapons/cannon.h"

#include "factory.h"
#include "key_mapper.h"
#include "event_handler.h"
#include "state.h"

#include "graphics_engine.h"

using namespace std;

CREATE_FACTORY(Hover, CL_HOVER);

/**
 *  destructs the hover, deletes alocated memory
 */
Hover::~Hover ()
{
}

/**
 * loads a Hover information from a specified file.
 * @param fileName the name of the File to load the hover from (absolute path)
 */
Hover::Hover(const char* fileName)
{
  this->init();
  TiXmlDocument doc(fileName);

  if(!doc.LoadFile())
  {
    PRINTF(2)("Loading file %s failed for Hover.\n", fileName);
    return;
  }

  this->loadParams(doc.RootElement());
}

/**
 *  creates a new Spaceship from Xml Data
 * @param root the xml element containing spaceship data

   @todo add more parameters to load
*/
Hover::Hover(const TiXmlElement* root)
{
  this->init();
  if (root != NULL)
    this->loadParams(root);

  //weapons:
  Weapon* wpRight = new TestGun(0);
  wpRight->setName("testGun Right");
  Weapon* wpLeft = new TestGun(1);
  wpLeft->setName("testGun Left");
  Weapon* cannon = dynamic_cast<Weapon*>(Factory::fabricate(CL_CANNON));

  cannon->setName("BFG");

  this->addWeapon(wpLeft, 1, 0);
  this->addWeapon(wpRight,1 ,1);
  this->addWeapon(cannon, 0, 2);

  this->getWeaponManager()->changeWeaponConfig(1);
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);

  this->loadModel("models/ships/hoverglider_mainbody.obj");
}


/**
 * initializes a Hover
 */
void Hover::init()
{
//  this->setRelDir(Quaternion(M_PI, Vector(1,0,0)));
  this->setClassID(CL_HOVER, "Hover");

  PRINTF(4)("HOVER INIT\n");

  this->loadModel("models/ships/hover_#.obj", 1.0);

  EventHandler::getInstance()->grabEvents(true);

  bUp = bDown = bLeft = bRight = bAscend = bDescend = bRollL = bRollR = false;
  bFire = false;
  xMouse = yMouse = 0;
  mouseSensitivity = 0.05;
  controlVelocityX = 100;
  controlVelocityY = 100;
  //rotorspeed = 1;
  //tailrotorspeed = 0;

  //cycle = 0.0;

  // cameraissue
  this->cameraNode.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->cameraNode.setParent(this);
  this->cameraNode.setParentMode(PNODE_ALL);


  // rotors
  this->wingNodeLeft.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT );
  this->wingNodeLeft.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);
  this->wingNodeLeft.setParent(this);
  this->wingNodeLeft.setRelCoor(-1.5, -.3, -1.0);
  this->rotorNodeLeft.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->rotorNodeLeft.setParent(&this->wingNodeLeft);
  this->rotorNodeLeft.setRelCoor(0, 1.0, -2.3);

  this->wingNodeRight.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->wingNodeRight.addNodeFlags(PNODE_PROHIBIT_CHILD_DELETE);
  this->wingNodeRight.setParent(this);
  this->wingNodeRight.setRelCoor(-1.5, -0.3, 1.0);
  this->rotorNodeRight.addNodeFlags(PNODE_PROHIBIT_DELETE_WITH_PARENT);
  this->rotorNodeRight.setParent(&this->wingNodeRight);
  this->rotorNodeRight.setRelCoor(0, 1.0, 2.3);

  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);

  this->loadModel("models/ships/hoverglider_wing.obj", 1.0f, 3);
  this->loadModel("models/ships/hoverglider_rotor.obj", 1.0f, 4);


  //travelSpeed = 15.0;
  this->velocity = Vector(0.0,0.0,0.0);
  this->velocityDir = Vector(1.0,0.0,0.0);

  //add events to the eventlist
  registerEvent(KeyMapper::PEV_UP);
  registerEvent(KeyMapper::PEV_DOWN);
  registerEvent(KeyMapper::PEV_LEFT);
  registerEvent(KeyMapper::PEV_RIGHT);
  //registerEvent(SDLK_q);
  registerEvent(SDLK_e);
  registerEvent(SDLK_c);
  registerEvent(KeyMapper::PEV_FIRE1);
  registerEvent(KeyMapper::PEV_NEXT_WEAPON);
  registerEvent(KeyMapper::PEV_PREVIOUS_WEAPON);
  //registerEvent(SDLK_PAGEUP);
  //registerEvent(SDLK_PAGEDOWN);
  registerEvent(EV_MOUSE_MOTION);


  // WEAPON_MANAGER configuration
  this->getWeaponManager()->setSlotCount(5);

  this->getWeaponManager()->setSlotPosition(0, Vector(-0.28, 1.186, -2.750), &this->wingNodeLeft);
  this->getWeaponManager()->setSlotCapability(0, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager()->setSlotPosition(1, Vector(-0.28, 1.186, 2.750), &this->wingNodeRight);
  this->getWeaponManager()->setSlotCapability(1, WTYPE_ALLDIRS | WTYPE_DIRECTIONAL);

  this->getWeaponManager()->setSlotPosition(2, Vector(-1.63, .809, -.003));
  this->getWeaponManager()->setSlotCapability(2, WTYPE_HEAVY);

  /// TODO: THESE ARE TOO MUCH
  this->getWeaponManager()->setSlotPosition(3, Vector(-1.63, .678, -.652));
  this->getWeaponManager()->setSlotDirection(3, Quaternion(-24/180 * M_PI, Vector(1,0,0)));

  this->getWeaponManager()->setSlotPosition(4, Vector(-1.63, .678, .652));
  this->getWeaponManager()->setSlotDirection(4, Quaternion(24/180 * M_PI, Vector(1,0,0)));

  this->getWeaponManager()->getFixedTarget()->setParent(&(this->cameraNode));
}

/**
 * loads the Settings of a Hover from an XML-element.
 * @param root the XML-element to load the Spaceship's properties from
 */
void Hover::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);
}

void  Hover::attachCamera()
{
  State::getCamera()->setParentSoft(&this->cameraNode);
  State::getCameraTarget()->setParentSoft(this->getWeaponManager()->getFixedTarget());

}


void Hover::enter()
{
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( true);
  this->attachCamera();


}

void Hover::leave()
{
  dynamic_cast<Element2D*>(this->getWeaponManager()->getFixedTarget())->setVisibility( false);
  this->detachCamera();

}


/**
 *  effect that occurs after the Hover is spawned
*/
void Hover::postSpawn ()
{
  //setCollision(new CollisionCluster(1.0, Vector(0,0,0)));
}

/**
 *  the action occuring if the hover left the game
*/
void Hover::leftWorld ()
{}

/**
 *  this function is called, when two entities collide
 * @param entity: the world entity with whom it collides
 *
 * Implement behaviour like damage application or other miscellaneous collision stuff in this function
 */
void Hover::collidesWith(WorldEntity* entity, const Vector& location)
{

}



/**
 *  the function called for each passing timeSnap
 * @param time The timespan passed since last update
*/
void Hover::tick (float dt)
{
  Playable::tick(dt);

  if( xMouse != 0 || yMouse != 0)
   {
    if (xMouse > controlVelocityX) xMouse = controlVelocityX;
    else if (xMouse < -controlVelocityX) xMouse = -controlVelocityX;
    if (yMouse > controlVelocityY) yMouse = controlVelocityY;
    else if (yMouse < -controlVelocityY) yMouse = -controlVelocityY;
  }

  // spaceship controlled movement
  this->calculateVelocity(dt);

  Vector move = (velocity)*dt;

  // this is the air friction (necessary for a smooth control)
  if(velocity.len() != 0) velocity -= velocity*0.1;
  this->shiftCoor (move);

}

/**
 *  calculate the velocity
 * @param time the timeslice since the last frame
*/
void Hover::calculateVelocity (float time)
{
  Vector accel(0.0, 0.0, 0.0);
  float rotValX = 0.0;
  float rotValZ = 0.0;
  /* FIXME: calculating the direction and orthDirection every timeSlice is redundant! save it somewhere */
  /* calculate the direction in which the craft is heading  */

  if( this->bUp )
   {
     //this->shiftCoor(this->getAbsDirX());
     //accel -= this->getAbsDirY();

     accel += Vector((this->getAbsDirX()).x,0,(this->getAbsDirX()).z);
     if((this->getAbsDirX()).y >= -0.1) rotValZ -= time;
   }
   else
   {
       if(this->getAbsDirX().y < -.02) this->shiftDir(Quaternion(time, Vector(0,0,1)));
   }

  if( this->bDown )
   {
     //this->shiftCoor((this->getAbsDirX())*-1);
     //accel -= this->getAbsDirY();

     accel -= Vector((this->getAbsDirX()).x,0,(this->getAbsDirX()).z);
     rotValZ += time;
   }
   else
   {
         if(this->getAbsDirX().y > 0.02) this->shiftDir(Quaternion(-time, Vector(0,0,1)));
   }

  if( this->bLeft /* > -this->getRelCoor().z*2*/)
  {
    //this->shiftDir(Quaternion(time, Vector(0,1,0)));
    //accel -= this->getAbsDirY();
    //velocityDir.normalize();

    accel -= Vector((this->getAbsDirZ()).x,0,(this->getAbsDirZ()).z);
    rotValX -= time;
  }
  else
   {
         if(this->getAbsDirZ().y > 0.02) this->shiftDir(Quaternion(time, Vector(1,0,0)));
   }

  if( this->bRight /* > this->getRelCoor().z*2*/)
  {
    //this->shiftDir(Quaternion(-time, Vector(0,1,0)));
    //accel += this->getAbsDirY();
    //velocityDir.normalize();

    accel += Vector((this->getAbsDirZ()).x,0,(this->getAbsDirZ()).z);
    rotValX += time;
  }
  else
   {
         if(this->getAbsDirZ().y < -0.02) this->shiftDir(Quaternion(-time, Vector(1,0,0)));
   }

  if( this->bRollL /* > -this->getRelCoor().z*2*/)
  {
    this->shiftDir(Quaternion(-time, Vector(1,0,0)));
    //accel -= rightDirection;
    //velocityDir.normalize();
    //rot +=Vector(1,0,0);
    //rotVal -= .4;
  }
  if( this->bRollR /* > this->getRelCoor().z*2*/)
  {
    this->shiftDir(Quaternion(time, Vector(1,0,0)));

    //accel += rightDirection;
    //velocityDir.normalize();
    //rot += Vector(1,0,0);
    //rotVal += .4;
  }
  if (this->bAscend )
  {
    //this->shiftDir(Quaternion(time, Vector(0,0,1)));

    accel += this->getAbsDirY();
    //rotorspeed += 0.05;
    //if (rotorspeed >= 2) rotorspeed = 2;
    //velocityDir.normalize();
    //rot += Vector(0,0,1);
    //rotVal += .4;
  }
  else
  {
    //if(rotorspeed >= 1.05) rotorspeed -= 0.05;
  }

  if (this->bDescend )
  {
    //this->shiftDir(Quaternion(-time, Vector(0,0,1)));

    accel -= this->getAbsDirY();
    //rotorspeed -= 0.05;
    //if (rotorspeed <= 0) rotorspeed = 0;
    //velocityDir.normalize();
    //rot += Vector(0,0,1);
    //rotVal -= .4;
  }
  else
  {
    //if(rotorspeed <= 0.05) rotorspeed += 0.05;
  }

  velocity += accel*3;
  if((this->getAbsDirX()).y <= 0.3 && (this->getAbsDirX()).y >= -0.3) this->shiftDir(Quaternion(rotValZ, Vector(0,0,1)));
  if((this->getAbsDirZ()).y <= 0.3 && (this->getAbsDirZ()).y >= -0.3) this->shiftDir(Quaternion(rotValX, Vector(1,0,0)));
}


void Hover::draw() const
{
  Vector tmpRot;
  WorldEntity::draw();

  glPushMatrix();
  /// LEFT SIDE
  glTranslatef (this->wingNodeLeft.getAbsCoor ().x,
                this->wingNodeLeft.getAbsCoor ().y,
                this->wingNodeLeft.getAbsCoor ().z);
  tmpRot = this->wingNodeLeft.getAbsDir().getSpacialAxis();
  glRotatef (this->wingNodeLeft.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  this->getModel(3)->draw();
  glPopMatrix ();

  glPushMatrix();
  glTranslatef (this->rotorNodeLeft.getAbsCoor ().x,
                this->rotorNodeLeft.getAbsCoor ().y,
                this->rotorNodeLeft.getAbsCoor ().z);
  tmpRot = this->rotorNodeLeft.getAbsDir().getSpacialAxis();
  glRotatef (this->rotorNodeLeft.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  this->getModel(4)->draw();
  glPopMatrix ();

  /// RIGHT SIDE
  glPushMatrix();
  glTranslatef (this->wingNodeRight.getAbsCoor ().x,
                this->wingNodeRight.getAbsCoor ().y,
                this->wingNodeRight.getAbsCoor ().z);
  tmpRot = this->wingNodeRight.getAbsDir().getSpacialAxis();
  glRotatef (this->wingNodeRight.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  glScalef(1,1,-1);
  this->getModel(3)->draw();
  glPopMatrix ();

  glPushMatrix();
  glTranslatef (this->rotorNodeRight.getAbsCoor ().x,
                this->rotorNodeRight.getAbsCoor ().y,
                this->rotorNodeRight.getAbsCoor ().z);
  tmpRot = this->rotorNodeRight.getAbsDir().getSpacialAxis();
  glRotatef (this->rotorNodeRight.getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );
  glScalef(1,1,-1);
  this->getModel(4)->draw();
  glPopMatrix ();

  this->getWeaponManager()->draw();
}


/**
 * weapon manipulation by the player
*/
void Hover::weaponAction()
{
  if( this->bFire)
    {
      this->getWeaponManager()->fire();
    }
}

/**
 * @todo switch statement ??
 */
void Hover::process(const Event &event)
{
  Playable::process(event);

  if( event.type == KeyMapper::PEV_LEFT)
      this->bLeft = event.bPressed;
  else if( event.type == KeyMapper::PEV_RIGHT)
      this->bRight = event.bPressed;
  else if( event.type == SDLK_e)
    this->bAscend = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == SDLK_c)
    this->bDescend = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == KeyMapper::PEV_UP)
    this->bUp = event.bPressed; //this->shiftCoor(0,.1,0);
  else if( event.type == KeyMapper::PEV_DOWN)
    this->bDown = event.bPressed; //this->shiftCoor(0,-.1,0);
  else if( event.type == EV_MOUSE_MOTION)
  {
    this->xMouse = event.xRel*mouseSensitivity;
    this->yMouse = event.yRel*mouseSensitivity;

    this->shiftDir(Quaternion(-M_PI/4*xMouse*mouseSensitivity, Vector(0,1,0)));

    Quaternion yDir = Quaternion(-M_PI/4*yMouse*mouseSensitivity, Vector(0,0,1));


    if ((this->cameraNode.getAbsDirY()).y < 0.5)
    {
     if((this->cameraNode.getAbsDirX()).y > 0)
     {
        if(yMouse > 0) this->cameraNode.shiftDir(yDir);
     }
     else
     {
         if(yMouse < 0) this->cameraNode.shiftDir(yDir);
     }
    }
    else this->cameraNode.shiftDir(yDir);;
    }
}
