/*!
 * @file weapon.h
 *
 * Weapon is the mayor baseclass for all weapons. it is quite extensive, and expensive in use,
 * because each weapon will know virutal functions for the WorldEntity's part, and also virtuals
 * for Fireing/Reloading/...,
 * quickly said: Weapon is a wrapper for weapons, that makes it easy to very quickly implement
 * new Weapons, and with them make this game better, than any game before it, because still
 * Weapons (GUNS) are the most important thing in life :?... no to be serious
 * @see Weapon
 */


#ifndef _WEAPON_H
#define _WEAPON_H

#include "world_entity.h"
#include "count_pointer.h"
#include "ammo_container.h"

// FORWARD DECLARATION
class Projectile;
class WeaponManager;
class Animation3D;
class TiXmlElement;
class FastFactory;
template<class T> class tFastFactory;
class GLGuiWidget;

//! An enumerator defining Actions a Weapon can take
typedef enum {
  WA_NONE          =    0,    //!< No Action taken
  WA_SHOOT         =    1,    //!< emitting Shot
  WA_CHARGE        =    2,    //!< charge action (one click before the shot)
  WA_RELOAD        =    3,    //!< reload right after shoot is finished
  WA_ACTIVATE      =    4,    //!< activate the GUN
  WA_DEACTIVATE    =    5,    //!< deactivate the GUN
  WA_SPECIAL1      =    6,    //!< Special Action taken

  WA_ACTION_COUNT  =    7     //!< This must match the count of enumerations-members.
} WeaponAction;

//! An enumerator defining the States of a Weapon
typedef enum {
  WS_NONE          =    0,    //!< No State at all (if set, there is something wrong, or the weapon is not yet availiable)
  WS_SHOOTING      =    1,    //!< The State of the Shooting
  WS_CHARGING      =    2,    //!< The state of charging th weapon
  WS_RELOADING     =    3,    //!< The State of the Reloading
  WS_ACTIVATING    =    4,    //!< The State in which the weapon gets activated
  WS_DEACTIVATING  =    5,    //!< The State in which the weapon gets deactivated
  WS_INACTIVE      =    6,    //!< The State where the weapon is inactive (unable to shoot)
  WS_IDLE          =    7,    //!< The State where the weapon is idle

  WS_STATE_COUNT  =     8     //!< This must match the count of enumerations-members.
} WeaponState;

//! an enumerator defining capabilities of a WeaponSlot
typedef enum
{
  WTYPE_DIRECTIONAL   = 0x00000001,           //!< Weapon is directional/Slot is able to carry directional weapons
  WTYPE_TURRET        = 0x00000002,           //!< Weapon is a turret/slot is able to carry turrets
  WTYPE_ALLKINDS      = 0x0000000f,           //!< Weapon is all types/Slot is able to carry all kinds of weapons

  WTYPE_FORWARD       = 0x00000010,           //!< Weapon fires forwards/Slot is able to carry weapons firing forwards
  WTYPE_BACKWARD      = 0x00000020,           //!< Weapon fires backwards/Slot is able to carry weapons firing backwards
  WTYPE_LEFT          = 0x00000040,           //!< Weapon fires to the left/Slot is able to carry weapons firing to the left
  WTYPE_RIGHT         = 0x00000080,           //!< Weapon fires to the right/Slot is able to carry weapons firing to the right
  WTYPE_ALLDIRS       = 0x000000f0,           //!< Weapon has no specific firing direction/Slot can fire into all directions

  WTYPE_ALL           = 0x000000ff,           //!< Weapon has no limitations/Slot can handle all kinds of Weapon.
} W_Capability;

//! An abstract class, that describes weapons
/**
 * This is used as a container for all the different kinds of weapons that may exist
 *
 * Weapons have certain states, and actions, that can inflict them.
 * ex. Action WA_SHOOT leeds to State WS_SHOOTING.
 * each action has a sound connected to it,
 * each state a time and an animation.
 */
class Weapon : public WorldEntity
{
  public:
    // INITIALISATION //
    Weapon ();
    virtual ~Weapon ();

    void init();
    virtual void loadParams(const TiXmlElement* root);
    ////////////////////

    // INTERACTIVITY //
    void requestAction(WeaponAction action);
    float increaseEnergy(float energyToAdd);
    ///////////////////

    /** @returns true if the Weapon is Active  (this is used to check if the weapon must be drawn)*/
    inline bool isActive() const { return (this->currentState == WS_INACTIVE)? false : true; };
    /** @returns true if the weapon must be drawn */
    inline bool isVisible() const { return (this->currentState != WS_INACTIVE || !this->hideInactive) ? true : false; };
    /** @returns true if the Weapon is chargeable */
    inline bool isChargeable() const { return this->chargeable; };

    // FUNCTIONS TO SET THE WEAPONS PROPERTIES.
    /** sets the Weapons Capabilities */
    inline void setCapability(long capabilities) { this->capability = capabilities; };
    /** @returns the Capabilities of this Weapon */
    inline long getCapability() const { return this->capability; };
    void setProjectileType(ClassID projectile);
    void setProjectileType(const char* projectile);
    /** @returns The projectile's classID */
    inline ClassID getProjectileType() { return this->projectile; };
    /** @returns the FastFactory, that creates Projectiles of type getProjectile */
    inline FastFactory* getProjectileFactory() { return this->projectileFactory; };
    void prepareProjectiles(unsigned int count);
    Projectile* getProjectile();


    // EMISSION
    void setEmissionPoint(const Vector& point);
    /** @see void setEmissionPoint(const Vector& point); */
    inline void setEmissionPoint(float x, float y, float z) { this->setEmissionPoint(Vector(x, y, z)); };
    /** @returns the absolute Position of the EmissionPoint */
    inline const Vector& getEmissionPoint() const { return this->emissionPoint.getAbsCoor(); };

    inline void setTarget(PNode* target) { this->target = target; };

    // STATE CHANGES //
    /** @param state the State to time @param duration the duration of the State */
    inline void setStateDuration(const char* state, float duration) { setStateDuration(charToState(state), duration); };
    /** @param state the State to time @param duration the duration of the State */
    inline void setStateDuration(WeaponState state, float duration) { /*(state < WS_STATE_COUNT)?*/this->times[state] = duration; };
    /** @param state The state to query @returns the Time the queried State takes to complete */
    inline float getStateDuration(WeaponState state) const { return (state < WS_STATE_COUNT)? this->times[state] : 0.0; };
    /** @returns true if the time of the currentState is elapsed, false otherwise */
    inline bool stateTimeElapsed() const { return (this->stateDuration > this->times[currentState])? true : false; };
    /** @returns the current State of the Weapon */
    inline WeaponState getCurrentState() const { return this->currentState; };

    /** @param energyMax the maximum energy the Weapon can have */
    inline void setEnergyMax(float energyMax) { this->energyMax = energyMax; };
    inline float getEnergy() const { return this->energy; };
    inline float getEnergyMax() const { return this->energyMax; };
    inline void setAmmoContainer(const CountPointer<AmmoContainer>& ammoContainer) { this->ammoContainer = ammoContainer;}

    void setActionSound(WeaponAction action, const char* soundFile);
    /** @see void setActionSound(WeaponAction action, const char* soundFile); */
    void setActionSound(const char* action, const char* soundFile) { this->setActionSound(charToAction(action), soundFile); };

    Animation3D* getAnimation(WeaponState state, PNode* node = NULL);
    Animation3D* copyAnimation(WeaponState from, WeaponState to);

    GLGuiWidget* getEnergyWidget();

    // FLOW
    bool tickW(float dt); //!< this is a function that must be called by the weaponManager, or any other weaponHandler, all other functions are handled from within

    virtual void tick(float dt) { tickW(dt); };

    bool check() const;
    void debug() const;

  protected:
    //! ACTION: these functions are handled by the Weapon itself, and must be called by requestAction(WeaponAction);
    virtual void activate() {};
    virtual void deactivate() {};
    virtual void charge() {};
    virtual void fire() {};
    virtual void reload() {};
    virtual void destroy() {};


    // utility:
    static WeaponAction  charToAction(const char* action);
    static const char*   actionToChar(WeaponAction action);
    static WeaponState   charToState(const char* state);
    static const char*   stateToChar(WeaponState state);

  private:
    /** executive functions, that handle timing with actions.
     * This is for the action-functions in derived functions to be easy
     * The main function is execute, that calls all the other functions
     * for being fast, the Functions are private and as such will be inlined
     * into the execute function. (this is work of the compiler)
     */
    bool execute();
    bool activateW();
    bool deactivateW();
    bool chargeW();
    bool fireW();
    bool reloadW();
    inline void enterState(WeaponState state);

    void updateWidgets();

  private:
    // type of Weapon
    long                 capability;                       //!< what capabilities the Weapon has @see W_Capability

    // it is all about energy
    float                energy;                           //!< The energy stored in the weapons buffers
    float                energyMax;                        //!< The maximal energy that can be stored in the secondary buffers (reserveMax)
    CountPointer<AmmoContainer> ammoContainer;             //!< Pointer to the AmmoContainer this weapon grabs Energy from.
    //! @todo move this to projectile
    float                minCharge;                        //!< The minimal energy to be loaded onto one projectile if chargeable otherwise the power consumed by one projectile
    float                maxCharge;                        //!< The maximal energy to be loaded onto one projectile (this is only availible if chargeable is enabled)

    GLGuiBar*            energyWidget;

    PNode*               target;                           //!< A target for targeting Weapons.

    ////////////
    // PHASES //
    ////////////
    SoundSource*         soundSource;                      //!< A SoundSource to play sound from (this is connected to the PNode of the Weapon)

    WeaponState          currentState;                     //!< The State the weapon is in.
    WeaponAction         requestedAction;                  //!< An action to try to Engage after the currentState ends.
    float                stateDuration;                    //!< how long the state has taken until now.
    float                times[WS_STATE_COUNT];            //!< Times to stay in the different States @see WeaponState.
    Animation3D*         animation[WS_STATE_COUNT];        //!< Animations for all the States (you can say yourself on what part of the gun this animation acts).
    SoundBuffer*         soundBuffers[WA_ACTION_COUNT];    //!< SoundBuffers for all actions @see WeaponAction.

    PNode                emissionPoint;                   //!< The point, where the projectiles are emitted. (this is coppled with the Weapon by default)

    bool                 hideInactive;                    //!< Hides the Weapon if it is inactive
    bool                 chargeable;                      //!< if the Weapon is charcheable (if true, the weapon will charge before it fires.)

    ClassID              projectile;                      //!< the projectile used for this weapon (since they should be generated via macro and the FastFactory, only the ClassID must be known.)
    FastFactory*         projectileFactory;               //!< A factory, that produces and handles the projectiles.
  };

#endif /* _WEAPON_H */
