/*!
 * @file entity_manager.h
 *  Manages creation and destruction of entities
 */

#ifndef _NETWORK_GAME_MANGER
#define _NETWORK_GAME_MANAGER

/* include this file, it contains some default definitions */
#include "netdefs.h"

/* include base_object.h since all classes are derived from this one */
#include "synchronizeable.h"


class TiXmlElement;
class PNode;

/**
 * protocol definition
 *
 *  CREATE_ENTITY:       >> CLASS_ID, UNIQUE_ID, OWNER
 *  REMOVE_ENTITY:       >> UNIQUE_ID
 *
 *  CREATE_ENTITY_LIST:  >> NUMBER, [CLASS_ID, UNIQUE_ID, OWNER][0..NUMBER]
 *  REMOVE_ENTITY_LIST:  >> NUMBER, [UNIQUE_ID][0..NUMBER]
 *
 *  REQUEST_CREATE:      >> CLASS_ID
 *                       << [Sync Info]
 *  REQUEST_REMOVE:      >> UNIQUE_ID
 *                       << [Sync Info]
 *
 *  //REQUEST_CREATE_LIST: NUMBER, [CLASS_ID][0..NUMBER]
 *  //REQUEST_CREATE_LIST: NUMBER, [UNIQUE_ID][0..NUMBER]
 *
 *  REQUEST_ENTITY_LIST: //request the whole world :D
 *  REQUEST_SYNC:        >> UNIQUE_ID
 *                       << [Sync Info]
 *  //REQUEST_SYNC_LIST:   NUMBER, [UNIQUE_ID][0..NUMBER]
 *
 *  YOU_ARE_ENTITY:      >> UNIQUE_ID
 *
 *  REQUEST_PNODE_PATH   >> UNIQUE_ID_START   UNIQUE_ID_STOP
 *                       << UNIQUE_ID_1   UNIQUE_ID_2   UNIQUE_ID_3 ...   UNIQUE_ID_N
 *
 *  SEND_PNODE_PATH      >> UNIQUE_ID_START   UNIQUE_ID_STOP NUMBER [UNIQUE_ID][0..NUMBER]
 */

typedef enum NetworkGameManagerProtocol {
  NET_CREATE_ENTITY = 0,
  NET_REMOVE_ENTITY,
  NET_CREATE_ENTITY_LIST,
  NET_REMOVE_ENTITY_LIST,
  NET_REQUEST_CREATE,
  NET_REQUEST_REMOVE,
  NET_REQUEST_SYNC,
  NET_YOU_ARE_ENTITY,
  NET_REQUEST_ENTITY_LIST,
  NET_REQUEST_PNODE_PATH,
  NET_SEND_PNODE_PATH,

  NET_NUMBER
};

struct clientBuffer
{
  int length;
  int maxLength;
  byte * buffer;
};

/*!
 * a class that can create and remove entities
 */
class NetworkGameManager: public Synchronizeable
{
  public:
    ~NetworkGameManager();

    static NetworkGameManager* NetworkGameManager::getInstance()
    { if (!NetworkGameManager::singletonRef) NetworkGameManager::singletonRef = new NetworkGameManager(); return NetworkGameManager::singletonRef; }

    virtual int writeBytes(const byte* data, int length, int sender);
    virtual int readBytes(byte* data, int maxLength, int * reciever);
    virtual void writeDebug() const;
    virtual void readDebug() const;

    int createEntity( ClassID classID, int owner = 0 );
    BaseObject* createEntity(const TiXmlElement* element);
    void removeEntity( int uniqueID );
    void sendYouAre( int uniqueID, int userID );

    void sync(int uniqueID, int owner);

    void sendEntityList(int userID);

    bool signalNewPlayer(int userId);
    bool signalLeftPlayer(int userID);


  private:
    NetworkGameManager();


    /* some network signal handlers */
    bool handleRequestCreate( int& i, const byte* data, int length, int sender );
    bool handleRequestRemove( int& i, const byte* data, int length, int sender );
    bool handleCreateEntity( int& i, const byte* data, int length, int sender );
    bool handleRemoveEntity( int& i, const byte* data, int length, int sender );
    bool handleCreateEntityList( int& i, const byte* data, int length, int sender );
    bool handleRemoveEntityList( int& i, const byte* data, int length, int sender );
    bool handleYouAreEntity( int& i, const byte* data, int length, int sender );
    bool handleRequestSync( int& i, const byte* data, int length, int sender );
    bool handleRequestPNodePath(int& i, const byte* data, int length, int sender);
    bool handleSendPNodePath(int& i, const byte* data, int length, int sender);


    /* some network handlers helper functions */
    void requestCreateEntity(ClassID classID);
    int executeCreateEntity(ClassID classID, int uniqueID = 0, int owner = 0);
    BaseObject* doCreateEntity(ClassID classID, int uniqueID, int owner);

    void requestRemoveEntity(int uniqueID);
    void executeRemoveEntity(int uniqueID);
    void doRemoveEntity(int uniqueID);

    void executeRequestSync( int uniqueID, int user );
    void doRequestSync(int uniqueID, int userID);

    void doYouAre( int uniqueID );

    void requestPNodePath(const PNode* node1, const PNode* node2);
    void executeRequestPNodePath(const PNode* node2, const PNode* node2);
    void doRequestPNodePath(const PNode* node1, const PNode* node2);

    bool canCreateEntity(ClassID classID);

    void resizeBufferVector(int n);

    bool writeToClientBuffer( clientBuffer &cb, byte*data, int length );
    bool writeToClientBuffer( clientBuffer &cb, byte b );
    bool writeToClientBuffer( clientBuffer &cb, int i );
    bool readFromClientBuffer( clientBuffer &cb, byte*data, int length );


  private:
    std::vector<clientBuffer>     outBuffer;
    //clientBuffer                  allOutBuffer;
    static NetworkGameManager*    singletonRef;
    bool                          hasRequestedWorld;
};

#endif /*_ENTITY_MANGER*/
