/*
   orxonox - the future of 3D-vertical-scrollers

   Copyright (C) 2004 orx

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   ### File Specific:
   main-programmer: Benjamin Grauer
   co-programmer: ...
*/
#define DEBUG_SPECIAL_MODULE DEBUG_MODULE_WORLD_ENTITY


#include "terrain.h"

#include "load_param.h"
#include "factory.h"
#include "spatial_separation.h"

#include "resource_manager.h"
#include "model.h"
#include "network_game_manager.h"


#include "glincl.h"

using namespace std;

CREATE_FACTORY(Terrain, CL_TERRAIN);

/**
 *  standard constructor
 */
Terrain::Terrain (const TiXmlElement* root)
{
  this->init();
  if( root != NULL)
    this->loadParams(root);

//  if (this->model != NULL)
    //this->ssp = new SpatialSeparation((Model*)this->model, 10.0f);
}


/**
 *  Constructor for loading a Terrain out of a file
 * @param fileName The file to load data from.

   this either loads out of an OBJ-file, or loads a heightmap if no .obj-extension is found.
*/
Terrain::Terrain(const char* fileName)
{
  this->init();

  if (!strstr(fileName, ".obj") || !strstr(fileName, ".OBJ") )
    {
      this->loadModel(fileName);
    }
  else
    {
      // load the hightMap here.
    }
}

/**
 *  a Constructor for the Debug-Worlds
 */
Terrain::Terrain(DebugTerrain debugTerrain)
{
  this->init();
  this->buildDebugTerrain(debugTerrain);
}

/**
 *  standard deconstructor

*/
Terrain::~Terrain ()
{
  if (objectList)
    glDeleteLists(this->objectList, 1);
  if( this->ssp)
    delete ssp;
  if (this->vegetation)
  {
    ResourceManager::getInstance()->unload(this->vegetation);
  }
}


void Terrain::init()
{
  this->setClassID(CL_TERRAIN, "Terrain");
  this->toList(OM_ENVIRON_NOTICK);

  this->objectList = 0;
  this->ssp = NULL;
  this->vegetation = NULL;
}


void Terrain::loadParams(const TiXmlElement* root)
{
  WorldEntity::loadParams(root);

  LoadParam(root, "vegetation", this, Terrain, loadVegetation)
      .describe("the fileName of the vegetation, that should be loaded onto this terrain. (must be relative to the data-dir)") ;
}

void Terrain::loadVegetation(const char* vegetationFile)
{
  PRINTF(0)("loadVegetation: %s\n", vegetationFile);
  if (this->vegetation)
    ResourceManager::getInstance()->unload(this->vegetation, RP_LEVEL);
  if (vegetationFile != NULL)
  {
    PRINTF(4)("fetching %s\n", vegetationFile);
    this->vegetation = (Model*)ResourceManager::getInstance()->load(vegetationFile, OBJ, RP_CAMPAIGN);
  }
  else
    this->vegetation = NULL;
}



void Terrain::draw () const
{
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();

  /* translate */
  glTranslatef (this->getAbsCoor ().x,
                this->getAbsCoor ().y,
                this->getAbsCoor ().z);
  /* rotate */
  Vector tmpRot = this->getAbsDir().getSpacialAxis();
  glRotatef (this->getAbsDir().getSpacialAxisAngle(), tmpRot.x, tmpRot.y, tmpRot.z );

  if (this->objectList)
    glCallList(this->objectList);
  else if (this->getModel())
    this->getModel()->draw();
  if (this->vegetation)
    this->vegetation->draw();
  glPopMatrix();

  /* THIS IS ONLY FOR DEBUGGING INFORMATION */
  if (this->ssp != NULL)
    this->ssp->drawQuadtree();
}


void Terrain::buildDebugTerrain(DebugTerrain debugTerrain)
{
  // if the terrain is the Terrain of Dave
  if (debugTerrain == TERRAIN_DAVE)
    {
      objectList = glGenLists(1);
      glNewList (objectList, GL_COMPILE);

      glColor3f(1.0,0,0);

      int sizeX = 100;
      int sizeZ = 80;
      float length = 1000;
      float width = 200;
      float widthX = float (length /sizeX);
      float widthZ = float (width /sizeZ);

      float height [sizeX][sizeZ];
      Vector normal_vectors[sizeX][sizeZ];


      for ( int i = 0; i<sizeX-1; i+=1)
        for (int j = 0; j<sizeZ-1;j+=1)
          //height[i][j] = rand()/20046 + (j-25)*(j-25)/30;
#ifdef __WIN32__
          height[i][j]=(sin((float)j/3)*rand()*i/182400)*.5;
#else
      height[i][j]=(sin((float)j/3)*rand()*(long)i/6282450500.0)*.5;
#endif

      //Die Huegel ein wenig glaetten
      for (int h=1; h<2;h++)
        for (int i=1;i<sizeX-2 ;i+=1 )
          for(int j=1;j<sizeZ-2;j+=1)
            height[i][j]=(height[i+1][j]+height[i][j+1]+height[i-1][j]+height[i][j-1])/4;

      //Berechnung von normalen Vektoren
      for(int i=1;i<sizeX-2;i+=1)
        for(int j=1;j<sizeZ-2 ;j+=1)
          {
            Vector v1 = Vector (widthX*(1),      height[i][j],      widthZ*(j) );
            Vector v2 = Vector (widthX*(i-1),    height[i-1][j],    widthZ*(j));
            Vector v3 = Vector (widthX*(i),      height[i][j+1],    widthZ*(j+1));
            Vector v4 = Vector (widthX*(i+1),    height[i+1][j],    widthZ*(j));
            Vector v5 = Vector (widthX*(i),      height[i][j-1],    widthZ*(j-1));

            Vector c1 = v2 - v1;
            Vector c2 = v3 - v1;
            Vector c3=  v4 - v1;
            Vector c4 = v5 - v1;
            Vector zero = Vector (0,0,0);
            normal_vectors[i][j]=c1.cross(v3-v5)+c2.cross(v4-v2)+c3.cross(v5-v3)+c4.cross(v2-v4);
            normal_vectors[i][j].normalize();
          }

      glBegin(GL_QUADS);
      int snowheight=3;
      for ( int i = 0; i<sizeX; i+=1)
        for (int j = 0; j<sizeZ;j+=1)
          {
            Vector v1 = Vector (widthX*(i),      height[i][j]-20,       widthZ*(j)  -width/2);
            Vector v2 = Vector (widthX*(i+1),    height[i+1][j]-20,     widthZ*(j)  -width/2);
            Vector v3 = Vector (widthX*(i+1),    height[i+1][j+1]-20,   widthZ*(j+1)-width/2);
            Vector v4 = Vector (widthX*(i),      height[i][j+1]-20,     widthZ*(j+1)-width/2);
            float a[3];
            if(height[i][j]<snowheight){
              a[0]=0;
              a[1]=1.0-height[i][j]/10-.3;
              a[2]=0;
              glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
            }
            else{
              a[0]=1.0;
              a[1]=1.0;
              a[2]=1.0;
              glMaterialfv(GL_FRONT,GL_DIFFUSE,a);

            }
            glNormal3f(normal_vectors[i][j].x, normal_vectors[i][j].y, normal_vectors[i][j].z);
            glVertex3f(v1.x, v1.y, v1.z);
            if(height[i+1][j]<snowheight){
              a[0]=0;
              a[1] =1.0-height[i+1][j]/10-.3;
              a[2]=0;
              glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
            }
            else{
              a[0]=1.0;
              a[1]=1.0;
              a[2]=1.0;
              glMaterialfv(GL_FRONT,GL_DIFFUSE,a);

            }
            glNormal3f(normal_vectors[i+1][j].x, normal_vectors[i+1][j].y, normal_vectors[i+1][j].z);
            glVertex3f(v2.x, v2.y, v2.z);
            if(height[i+1][j+1]<snowheight){
              a[0]=0;
              a[1] =1.0-height[i+1][j+1]/10-.3;
              a[2]=0;
              glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
            }
            else{
              a[0]=1.0;
              a[1]=1.0;
              a[2]=1.0;
              glMaterialfv(GL_FRONT,GL_DIFFUSE,a);


            }
            glNormal3f(normal_vectors[i+1][j+1].x, normal_vectors[i+1][j+1].y, normal_vectors[i+1][j+1].z);
            glVertex3f(v3.x, v3.y, v3.z);
            if(height[i][j+1]<snowheight){
              a[0]=0;
              a[1] =1.0-height[i+1][j+1]/10-.3;
              a[2]=0;
              glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
            }
            else{
              a[0]=1.0;
              a[1]=1.0;
              a[2]=1.0;
              glMaterialfv(GL_FRONT,GL_DIFFUSE,a);
            }
            glNormal3f(normal_vectors[i][j+1].x, normal_vectors[i][j+1].y, normal_vectors[i][j+1].z);
            glVertex3f(v4.x, v4.y, v4.z);

          }
      glEnd();
      glEndList();
    }

  if (debugTerrain == TERRAIN_BENSCH)
    {
      /*
        this->model = (OBJModel*) new Model();
      this->model->setName("CUBE");
      this->model->addVertex (-0.5, -0.5, 0.5);
      this->model->addVertex (0.5, -0.5, 0.5);
      this->model->addVertex (-0.5, 0.5, 0.5);
      this->model->addVertex (0.5, 0.5, 0.5);
      this->model->addVertex (-0.5, 0.5, -0.5);
      this->model->addVertex (0.5, 0.5, -0.5);
      this->model->addVertex (-0.5, -0.5, -0.5);
      this->model->addVertex (0.5, -0.5, -0.5);

      this->model->addVertexTexture (0.0, 0.0);
      this->model->addVertexTexture (1.0, 0.0);
      this->model->addVertexTexture (0.0, 1.0);
      this->model->addVertexTexture (1.0, 1.0);
      this->model->addVertexTexture (0.0, 2.0);
      this->model->addVertexTexture (1.0, 2.0);
      this->model->addVertexTexture (0.0, 3.0);
      this->model->addVertexTexture (1.0, 3.0);
      this->model->addVertexTexture (0.0, 4.0);
      this->model->addVertexTexture (1.0, 4.0);
      this->model->addVertexTexture (2.0, 0.0);
      this->model->addVertexTexture (2.0, 1.0);
      this->model->addVertexTexture (-1.0, 0.0);
      this->model->addVertexTexture (-1.0, 1.0);

      this->model->finalize();
      */
    }
}

int Terrain::writeBytes( const byte * data, int length, int sender )
{
  setRequestedSync( false );
  setIsOutOfSync( false );

  SYNCHELP_READ_BEGIN();
  SYNCHELP_READ_FKT( WorldEntity::writeState );

  return SYNCHELP_READ_N;
}

int Terrain::readBytes( byte * data, int maxLength, int * reciever )
{
  if ( isOutOfSync() && !requestedSync() && this->getHostID()!=this->getOwner() )
  {
    (NetworkGameManager::getInstance())->sync( this->getUniqueID(), this->getOwner() );
    setRequestedSync( true );
  }

  int rec = this->getRequestSync();
  if ( rec > 0 )
  {
    *reciever = rec;

    return WorldEntity::readState( data, maxLength );

  }

  *reciever = 0;
  return 0;
}

void Terrain::writeDebug( ) const
{
}

void Terrain::readDebug( ) const
{
}
